import logging
import os
import re
import subprocess
import uuid
import zipfile

from .exceptions import InvalidUploadedFileError, InvalidAudioFileError
from .stored_files import checkIfFileExtensionIsAccepted
from .system_info import getStorageSize
from .utils import readFromJSONFile, writeToJSONFile

log = logging.getLogger('flask-backend')

'''
Get some details from audio files: filename, duration and size.
@param fileList: list of dictionaries, where each dictionary contains the directory where the file is and the file itself.
Returns a list of dictionaries, where each dictionary contains the details requested.
'''
def getAudioFileDetails(fileList):
    infoFileList = []
    for elem in fileList:
        fileInfo = {
            "filename": elem['file'].name,
            "duration": getAudioFileDuration(os.path.join(elem['dir'], elem['file'].name)),
            "size": elem['file'].stat().st_size
        }
        infoFileList.append(fileInfo)
    return infoFileList


'''
Upload an Audio File and store it in the device.
@param fileUploaded: audio file uploaded
@param destinationPath: string containing the full path of where the file should be stored
@param fileFormatsAllowed: list of strings containing the audio files extensions allowed
@param freeSpace: int representing the free space left on the device
@param audioFilesJSONFilePath: string containing the full path of the file that will store the information about the audio file uploaded.
Validates if the device has enough space to store the audio file. If not, raises an exception.
Validates if the file uploaded has a valid extension. It can be a zip file as well. If the file is invalid, or any file inside of the zip file, return an InvalidUploadedFileError exception.
If everything goes well, the file is stored in the JSON audio file as well.
Returns the number of audio files stored.
If an exception occurs after the file has been stored, remove the file.
'''
def uploadAudioFile(fileUploaded, destinationPath, fileFormatsAllowed, freeSpace, audioFilesJSONFilePath):
    destinationFilePath = ""
    uploadedFileIsZip = False
    try:
        # generate ID to be the name of the file when stored
        fileID = uuid.uuid4().hex
        destinationFilePath = os.path.join(destinationPath, fileID)
        # save file in right directory
        fileUploaded.save(destinationFilePath)
        # check if file size is larger than available free space in storage (because of 5% root dedicated space, file is stored even if available space shows as 0, since backed is being run as root)
        if os.stat(destinationFilePath).st_size > freeSpace:
            raise Exception('No space left on device')
        # check if file is zip
        uploadedFileIsZip = zipfile.is_zipfile(destinationFilePath)
        if uploadedFileIsZip:
            nFilesStored = unzipAudioFiles(destinationFilePath, destinationPath, fileFormatsAllowed, freeSpace)
        else:
            checkIfFileExtensionIsAccepted(fileUploaded.filename, fileFormatsAllowed)
            storeAudioFile(fileID, fileUploaded.filename, destinationFilePath, audioFilesJSONFilePath)
            nFilesStored = 1
    except Exception as e:
        log.error(e)
        # if file was stored and then some error occurred, file should be removed
        if destinationFilePath != "" and os.path.exists(destinationFilePath):
            os.remove(destinationFilePath)
        if uploadedFileIsZip:
            raise InvalidUploadedFileError("Invalid zip file's content")
        else:
            raise e
    return nFilesStored


'''
Save the file information in the JSON audio file.
@param fileID: id with which the file was stored in the device (name of the file stored)
@param filename: name associated with the file
@param fileLocation: sting containing the full path of the file
@param audioFilesJSONFilePath: string containing the full path of the JSON file that will store the information about the file provided.
The audioFilesJSONFilePath contains a list of dictionaries, each containing the following information regarding an audio file:
    - file_id: id with which the file was stored in the device (name of the file stored)
    - filename: name associated with the file
    - notes: some string containing a some description or notes
    - duration: string containing the duration of the audio file
    - size: size of the audio file in bytes
'''
def storeAudioFile(fileID, filename, fileLocation, audioFilesJSONFilePath):
    try:
        # create file info dictionary entry to store
        fileInfo = {
            "file_id": fileID,
            "filename": filename,
            "notes": "",
            "duration": getAudioFileDuration(fileLocation),
            "size": os.path.getsize(fileLocation)
        }
        # update JSON file with stored audio files information
        if os.path.isfile(audioFilesJSONFilePath):
            currentFileDetailsList = readFromJSONFile(audioFilesJSONFilePath)
        else:
            currentFileDetailsList = []
        currentFileDetailsList.append(fileInfo)
        writeToJSONFile(audioFilesJSONFilePath, currentFileDetailsList)
    except Exception as e:
        log.error(e)
        raise e


'''
Unzip the zip file provided and store the audio files inside in the device.
@param zipfilePath: string containing the full path of the zip file
@param destinationPath: string containing the full path of the directory to where the files should be unzipped
@param fileFormatsAllowed: list of strings containing the audio files extensions allowed
@param audioFilesJSONFilePath: string containing the full path of the JSON file that will store the information about the audio files unzipped
The files are unzipped, one by one, to the destinationPath. For each file:
 - Validates if the device has enough space to store the audio file. If not, raises an exception.
 - Validates if the file unzipped has a valid extension. If the file is invalid, returns an InvalidUploadedFileError exception.
 - If everything goes well, the file is stored in the JSON audio file.
Returns the number of audio files stored.
If an exception occurs during the process, remove all the files that were unzipped so far.
'''
def unzipAudioFiles(zipfilePath, destinationPath, fileFormatsAllowed, audioFilesJSONFilePath):
    filesStored = []
    try:
        with zipfile.ZipFile(zipfilePath, 'r') as fzip:
            for entry in fzip.infolist():
                filename = entry.filename
                if not filename.endswith('/'): # ignore directories, focus only in files
                    # generate ID to be the name of the file when stored
                    fileID = uuid.uuid4().hex
                    destinationFilePath = os.path.join(destinationPath, fileID)
                    entry.filename = fileID # store file with ID. not name
                    # get free space size in storage
                    freeSpace = getStorageSize()["available"]
                    fzip.extract(entry, destinationPath) # extract and save file in right place
                    # check if file size is larger than available free space in storage (because of 5% root dedicated space)
                    if os.stat(destinationFilePath).st_size > freeSpace:
                        raise Exception('No space left on device')
                    checkIfFileExtensionIsAccepted(filename, fileFormatsAllowed)
                    storeAudioFile(fileID, filename, destinationFilePath, audioFilesJSONFilePath)
                    filesStored.append(destinationFilePath)
    except Exception as e:
        log.error(e)
        os.remove(destinationFilePath)
        removeAudioFiles(audioFilesJSONFilePath, filesStored)
        raise e
    else:
        try:
            os.remove(zipfilePath)
        except Exception as e:
            log.error(e)
            raise e
        else:
            return len(filesStored)


'''
Get the duration of an audio file provided
@param filePath: string containing the full path of the audio file
Returns a string in the format 'HH:mm:ss' representing the duration of the file
If an error occurs during the process (for example when the file provided is not a valid audio file), an InvalidAudioFileError Exception is raised.
'''
def getAudioFileDuration(filePath):
    try:
        process = subprocess.Popen(['ffmpeg', '-i', filePath], stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
        stdout, stderr = process.communicate()
        matches = re.search(r"Duration:\s{1}(?P<duration>\d+:\d+:\d+?)(\.\d+?),", stdout.decode(),re.DOTALL).groupdict()
        fileDuration = matches['duration']
    except Exception:
        log.error("{} is an invalid Audio File".format(filePath))
        raise InvalidAudioFileError
    else:
        return fileDuration


'''
Removes the files, present in the list provided, from the device and from the JSON file
@param audioFilesJSONFilePath: string containing the full path of the JSON file that contains the information about the audio files uploaded to the device
@param filesList: list containing the path of the files to remove
Returns the number of audio files removed
'''
def removeAudioFiles(audioFilesJSONFilePath, filesList):
    try:
        nAudioFilesRemoved = 0
        if os.path.isfile(audioFilesJSONFilePath):
            currentAudioFilesList = readFromJSONFile(audioFilesJSONFilePath)
        else:
            currentAudioFilesList = []
        for filePath in filesList:
            for fileInfo in currentAudioFilesList:
                if os.path.isfile(filePath): # check if file exists
                    if filePath.split('/')[-1] == fileInfo["file_id"]:
                        currentAudioFilesList.remove(fileInfo)
                        os.remove(filePath)
                        log.debug("File {} removed".format(filePath))
                        nAudioFilesRemoved += 1
                else:
                    currentAudioFilesList.remove(fileInfo)
                    log.debug("File {} removed from audio JSON file".format(fileInfo["file_id"]))
        writeToJSONFile(audioFilesJSONFilePath, currentAudioFilesList)
    except Exception as e:
        log.error(e)
        raise e
    else:
        return nAudioFilesRemoved