#  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#  [21/5/2024] Created by Anton Angov for Barix AG.
#  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
from os import system

from barix.system.barix_enums import BarixHWType, BarixReturn


class BarixSystemFunctions:
    """
        Only tested for LX400 and M400 devices.
    """
    def __init__(self, hw_type, fw_ver):
        self.hw_type = hw_type
        self.eth_proc = 0
        if BarixHWType(self.hw_type) == BarixHWType.LX400:
            self.eth_proc = 45
        elif BarixHWType(self.hw_type) == BarixHWType.Barix_M400:
            self.eth_proc = 36
        self.fw_ver = fw_ver
        self._tmpfs_mount_points = ["/var/volatile", "/run"]

    def set_tmpfs_partition_size(self, new_size: int, unit: str, mount_point: str) -> bool:
        """
            Set the tmpfs in MB or % from the total available.
            It needs to be used separately for each partition,
            in order to keep the allocated memory consistent with the available memory
        @param new_size: New size of the allocated memory
        @param unit: M or %
        @param mount_point: /run or /var/volatile
        @return: True or False
        """

        if mount_point in self._tmpfs_mount_points:
            command = f"mount -o remount,size={new_size}{unit} {mount_point}"
            result = system(command)
            if result == 0:
                print(f"Successfully resized {mount_point} to {new_size}{unit}")
                return True

            print(f"Failed to resize {mount_point} to {new_size}{unit}")
            return False
        else:
            print(f"Mount point unknown: {mount_point}")
            return False

    def eth_affinity(self, proc: int = None, new_affinity: str = 'f', get: bool = False) -> str:
        """
            By default the ethernet adapter interrupts are processed by CPU0.
            However, it happens so that CPU0 is the one responsible for processing most operations,
            while the rest of the CPUs are hardly used.
            Changing the affinity on the device might improve performance.
        @param proc: Which processes affinity to change
        @param new_affinity: New affinity value.
                Valid values are f, e, c and 8
        @param get: If only reading set to True, otherwise the affinity will be set
        @return: Result of the operation
        """
        if proc is None:
            proc = self.eth_proc

        if not proc:
            msg = 'No proc to change affinity. Do nothing'
            print(msg)
            return msg

        typical_values = ['f', 'e', 'c', '8']
        if new_affinity not in typical_values:
            print(f"Affinity value {new_affinity} not typical (e.g. {typical_values}). "
                  f"Make sure it provides the desired result")

        print(f"Setting proc: {proc} affinity to: {new_affinity}")
        smp_affinity = f"/proc/irq/{proc}/smp_affinity"

        if get:
            with open(smp_affinity, 'r') as eth:
                result = eth.read()
        else:
            with open(smp_affinity, 'w') as eth:
                result = eth.writelines(new_affinity)
        return result
