import os
import json
import socket
import struct
import fcntl
import logging
import subprocess
import ifaddr
import ipaddress

SIOCGIFADDR = 0x8915
# Is must be a bytes object to be used by struct.pack
IFNAME = b'eth0'

logger = logging.getLogger(__name__)

def get_eth_address():
    address=''
    sckt = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    try:
        ifreq = struct.pack('256s', IFNAME[:15])
        res = fcntl.ioctl(sckt.fileno(), SIOCGIFADDR, ifreq)
        address = socket.inet_ntoa(res[20:24])
    except Exception as err:
        logger.error('Error getting ethernet address: {}'.format(err))
        raise err

    return address

def get_broadcast_address(interface='eth0'):
    adapters = ifaddr.get_adapters()
    for adapter in adapters:
        if adapter.nice_name == interface:
            for ip in adapter.ips:
                if ip.is_IPv4:
                    network = ipaddress.IPv4Network(f"{ip.ip}/{ip.network_prefix}", strict=False)
                    return str(network.broadcast_address)
    return None

def get_firmware_version():
    version = ''
    with open('/barix/info/VERSION', 'r') as f:
        version = f.read().rsplit()[0]
    return version

def get_hw_type() -> str:
    hwType = ''
    try:
        # qiba-spi-get-production-info.sh always returns an error
        # Valid on bavidi
        if os.path.isfile('/usr/bin/bavidi-read-prod-info.sh'):
            all_info = subprocess.check_output('/usr/bin/bavidi-read-prod-info.sh', shell=True, stderr=subprocess.STDOUT)
            decoded_dict = json.loads(str(all_info.decode('utf-8')).replace('\n', ''))
            hwType = decoded_dict["Production_Info"]["Hw_Type"]

        # Valid on flexa, paging master and others
        elif os.path.isfile('/usr/bin/qiba-spi-get-production-info.sh'):
            hwType = subprocess.check_output('/usr/bin/qiba-spi-get-production-info.sh -w', shell=True, stderr=subprocess.STDOUT)

    except subprocess.CalledProcessError as e:
        hwType = str(e.output.decode('utf-8')).replace('\n', '')
    except Exception as err:
        logger.error("Error getting hw type: {}".format(err))
        raise err

    return hwType

def get_eth_mac():
    mac = ''
    try:
        output = subprocess.check_output(['/usr/bin/qiba-spi-get-mac-address'])
        for line in output.splitlines():
            key, value = line.decode().split("=", 1)
            if key == 'ethaddr':
                mac = value
                break
    except subprocess.CalledProcessError as cmdError:
        logger.error('{} returned error code {}'.format(cmdError.cmd, cmd.returncode))
        raise cmdError

    return mac

def get_cpu_snr():
    snr = ''
    try:
        output = subprocess.check_output(['/usr/bin/qiba-get-params.sh'])
        for line in output.splitlines():
            key, value = line.decode().split("=", 1)
            if key == 'serialno':
                snr = value
                break
    except subprocess.CalledProcessError as cmdError:
        logger.error('{} returned error code {}'.format(cmdError.cmd, cmd.returncode))
        raise cmdError

    return snr
    

def get_cpu_temp():
    """
    Retrieve the CPU temperature in degrees Celsius.
    If is not supported, None is returned.
    """
    temperature = None
    with open('/sys/class/thermal/thermal_zone0/temp', "r") as f:
        temp = int(f.read())
        temperature = temp/1000.0
    return temperature
