import logging
import os
import pjsua as pj
from threading import *
import time
import pkg_resources
from enum import Enum
from . AudioSource import *


class SipAccountEvent:
    def __init__(self):
        pass
    
    REGISTERED = 1
    UNREGISTERED = 2
    INCOMMING_CALL = 3
    
class SipCallEvent:
    def __init__(self):
        pass
    
    CALL_DISCONNECT  = 11
    MEDIA_ACTIVE     = 12
    MEDIA_INACTIVE   = 13
    CALL_CONFIRMED   = 14
    CALL_RINGING     = 15
    CALL_CONNECTING  = 16
    CALL_CALLING     = 17
    CALL_BUSY        = 18

class SipCallState(Enum):
    IDLE            = 0
    CALLING         = 1
    INCOMMING_CALL  = 2
    RINGING         = 3
    CALLING_RINGING = 4
    CONNECTED       = 5
    GETTING_NOTIF   = 6

class HalfDuplexState(Enum):
    TALKING         = 0
    LISTENING       = 1


# Logging callback
def log_cb(level, str, len):
    print(str)

class SipClient(AudioSource):

    BUSY_TONE_PLAY_TIME = 3 # time in seconds for playing busy tone

    def __init__(self, sourceId, name, volume, parameters, audioDev, capDev='plug:dsnooper'):
        AudioSource.__init__(self, sourceId, name, volume, parameters, audioDev)

        self.sip_mode = str(AudioSource.getParameter(self,"sip_mode"))
        if self.sip_mode != "peer_to_peer":
            self.domain = str(AudioSource.getParameter(self,"domain"))
            self.username = str(AudioSource.getParameter(self,"user_name"))
            self.password = str(AudioSource.getParameter(self,"password"))
            self.transport_and_encryption = str(AudioSource.getParameter(self,"transport_and_encryption"))

        self.sndRecLatency = AudioSource.getParameter(self,"snd_rec_latency")
        self.sndPlayLatency = AudioSource.getParameter(self,"snd_play_latency")
        self.sndClockRate = AudioSource.getParameter(self,"snd_clock_rate")
        self.channelCount = AudioSource.getParameter(self,"channel_count")
        self.jitterMax = AudioSource.getParameter(self,"jb_max")

        self.autoAnswer = str(AudioSource.getParameter(self,"auto_answer"))
        if self.autoAnswer == "true":
            self.autoAnswer = True
        else:
            self.autoAnswer = False

        self.timeToAnswer   = int(AudioSource.getParameter(self,"answer_time"))
        self.callTimeout    = int(AudioSource.getParameter(self,"call_timeout"))
        self.streamTimeout  = int(AudioSource.getParameter(self,"stream_timeout"))
        self.beepOnAnswer   = str(AudioSource.getParameter(self,"beep_on_answer"))
        if self.beepOnAnswer == "true":
            self.beepOnAnswer = True
        else:
            self.beepOnAnswer = False
        
        self.halfDuplex     = str(AudioSource.getParameter(self,"half_duplex"))
        if self.halfDuplex == "true":
            self.halfDuplex = True
            self.halfDuplexMode = HalfDuplexState.TALKING
        else:
            self.halfDuplex = False

        self.audioEchoCancel = str(AudioSource.getParameter(self,"aec"))
        if self.audioEchoCancel == "true":
            self.audioEchoCancel = True
        else:
            self.audioEchoCancel = False
            
        self.capDev             = capDev

        self.inputGain              = float(AudioSource.getParameter(self,"input_gain"))
        self.hdxLevel               = float(AudioSource.getParameter(self,"hdx_level"))
        self.hdxTimeout             = float(AudioSource.getParameter(self,"hdx_timeout"))

        self.lib = None

        self.acc                    = None
        self.current_call           = None
        self.on_call                = False
        self.registered             = False
        self.call_slot              = None

        self.lock                   = Lock()

        self.state                  = SourceState.STOPPED
        self.alive                  = False
        self.counter                = 0
        self.sipCallState           = SipCallState.IDLE
        self.isWaitingToAnswer      = False

        self.incommingCallReceived  = False

        self.requestToAnswerCall        = False
        self.requestToHangupCall        = False
        self.requestToMakeCall          = False
        self.requestCheckCallTimeout    = False
        self.requestCheckStreamTimeout  = False
        self.requestToConnectCall       = False
        self.requestCheckBusyTone       = False
        self.requestCheckHalfDuplex     = False
        self.requestToDisconnect        = False
        self.requestToDisconnectPJSUA   = False
        self.regTimeout                 = 30
        self.regRetryInterval           = 30
        self.requestToRegister          = False    

        self.dstSipUri              = None

        self.waitTone               = pkg_resources.resource_filename('barix.resources', 'wait-tone.wav')
        self.ringTone               = pkg_resources.resource_filename('barix.resources', 'ring-tone.wav')
        self.busyTone               = pkg_resources.resource_filename('barix.resources', 'busy-tone.wav')
        self.beepTone               = pkg_resources.resource_filename('barix.resources', 'beep-tone.wav')
        self.wavPlayerId            = None
        self.wavPlayerSlot          = None

        self.currentCall_id         = None
        self.reportTransSecure      = "No"

        self.reportSipResponseCode  = ""

        self.logger                 = logging.getLogger('sip')

        self.requestedState         = None
        self.terminate              = False

        self.sip_thread             = Thread(target=self.workerThread)
        self.sip_thread.daemon      = True
        self.sip_thread.start()

    def workerThread(self):
        self.lib = pj.Lib()

        media = pj.MediaConfig()
        
        if self.audioEchoCancel:
            self.logger.info("Audio Echo Cancellation Enabled")
            media.ec_options  = 1   # use Speex AEC algorithm
            media.ec_tail_len = 256 # ms
        else:
            self.logger.info("Audio Echo Cancellation Disabled")
            media.ec_options  = 0   # default AEC mode
            media.ec_tail_len = 0   # disable Echo Canceller

        audioDev    = super().getAudioDev()
        self.logger.info("Playback Dev:{}, Capture Dev:{}".format(audioDev,self.capDev))
        media.pcmp_dev_name = audioDev
        media.pcmc_dev_name = self.capDev
        if self.sndRecLatency is not None:
            media.snd_rec_latency = int(self.sndRecLatency)
        if self.sndPlayLatency is not None:
            media.snd_play_latency = int(self.sndPlayLatency)
        if self.sndClockRate is not None:
            media.snd_clock_rate = int(self.sndClockRate)
        if self.channelCount is not None:
            media.channel_count = int(self.channelCount)
        if self.jitterMax is not None:
            media.jb_max = int(self.jitterMax)

        # Init library with default config and some customized logging config.
        self.lib.init(log_cfg=pj.LogConfig(level=3), media_cfg=media)

        if self.sip_mode != "peer_to_peer":
            # Transport Initialization needed if there is a SIP Server
            if self.transport_and_encryption == "udp":
                # Create UDP transport which listens to any available port
                self.lib.create_transport(pj.TransportType.UDP, pj.TransportConfig(0))
                self.logger.info("SIP CLIENT: Transport->UDP")
            elif self.transport_and_encryption == "tcp":
                self.lib.create_transport(pj.TransportType.TCP, pj.TransportConfig(0))
                self.logger.info("SIP CLIENT: Transport->TCP")
            else:
                self.lib.create_transport(pj.TransportType.TLS, pj.TransportConfig(0))
                self.logger.info("SIP CLIENT: Transport->TLS")
        # Start the library
        self.lib.start()

        if self.sip_mode != "server":
            # Setup local Account to allow SIP Peer-To-Peer calls
            transport1 = self.lib.create_transport(pj.TransportType.UDP,pj.TransportConfig(5060))
            self.lib.create_account_for_transport(transport1, cb=MyAccountCallback(None, self))
            self.logger.info("SIP CLIENT: Setup Local Account for Peer-to-peer calls :Transport->UDP")

        if self.sip_mode != "peer_to_peer":
            self.connect()

        while not self.terminate:
            with self.lock:
                if self.requestCheckHalfDuplex:
                    self.processHalfDuplex()
                if self.requestCheckCallTimeout:
                    self.checkCallTimeout()
                if self.requestCheckStreamTimeout:
                    self.checkStreamTimeout()
                if self.requestToConnectCall:
                    self.connectCallState()
                    self.requestToConnectCall = False
                if self.requestToDisconnect:
                    self.disconnectCallState()
                if self.requestToHangupCall:
                    self.hangup()
                if self.requestToAnswerCall:
                    self.answer()
                if self.requestToMakeCall:
                    self.call()
                if self.isWaitingToAnswer:
                    self.checkTimeToAnswer()
                if self.requestCheckBusyTone:
                    self.checkBusyTone()
                if self.requestToRegister:
                    self.restartRegistration()
                if self.requestToDisconnectPJSUA:
                    self.disconnectPJSUA()
                    return
            time.sleep(0.01)
    
    def connect(self):
        # register device on configurable SIP SERVER
        if self.transport_and_encryption == "udp":
            accountConfig = pj.AccountConfig(str(self.domain),str(self.username),str(self.password))
            self.logger.info("SIP CLIENT: Transport->UDP, No Media Encryption")
        elif self.transport_and_encryption == 'tcp':
            proxy = "sip:" + str(self.domain) + ";transport=tcp"
            accountConfig = pj.AccountConfig(str(self.domain), str(self.username), str(self.password),"","",proxy)
            self.logger.info("SIP CLIENT: Transport->TCP, No Media Encryption")
        elif self.transport_and_encryption == "tls":
            proxy = "sip:" + str(self.domain) + ";transport=tls"
            accountConfig = pj.AccountConfig(str(self.domain), str(self.username), str(self.password),"","",proxy)
            self.logger.info("SIP CLIENT: Transport->TLS, No Media Encryption")
            self.reportTransSecure = "SIP secure (TLS)"
        else: #tls + srtp (2=enable; 1=hop-by-hop secure transport such as TLS is required
            proxy = "sip:" + str(self.domain) + ";transport=tls"
            accountConfig = pj.AccountConfig(str(self.domain), str(self.username), str(self.password), "", "", proxy, 2, 1)
            self.logger.info("SIP CLIENT: Transport->TLS, Media Encryption -> SRTP")
            self.reportTransSecure = "SIP secure (TLS) and Media Encryption (SRTP)"

        accountConfig.reg_timeout          = self.regTimeout
        accountConfig.reg_retry_interval   = self.regRetryInterval 
        self.acc = self.lib.create_account(accountConfig)
        # Set Account callback
        acc_cb = MyAccountCallback(self.acc, self)
        self.acc.set_callback(acc_cb)

    def playAudio(self):
        self.counter += 1
        self.logger.info("SIP CLIENT source id={}, state: PLAY, \"{}\"".format(self.id, self.name))
        self.state = SourceState.PLAYING
        AudioSource.mute(self, False)

    def stopAudio(self):
        self.logger.info("SIP CLIENT source id={}, state: STOP, \"{}\"".format(self.id, self.name))
        self.state = SourceState.STOPPED
        self.logger.warning("IMPLEMENT ME!!")
        AudioSource.mute(self, True)

    def checkAudio(self):
        self.logger.info("SIP CLIENT source id={}, state: MONITOR, \"{}\"".format(self.id, self.name))
        self.state = SourceState.MONITORING
        AudioSource.mute(self, True)
    
    def getState(self):
        return self.state , self.alive

    def answerCall(self):
        self.requestToAnswerCall = True

    def hangupCall(self):
        self.requestToHangupCall = True

    def makeCall(self, dst_sip_id):
        if self.sip_mode != "peer_to_peer":
            self.dstSipUri = "sip:{}@{}".format(dst_sip_id,self.domain)
            self.logger.info("Will call {}".format(self.dstSipUri))
            self.requestToMakeCall = True

    def checkTimeToAnswer(self):
        if ( time.monotonic() - self.startTimeAnswer ) >= self.timeToAnswer:
            self.logger.info("waiting to answer timeout of {} seconds reached, will answer call".format(self.timeToAnswer))
            self.answer()
            self.isWaitingToAnswer = False

    def answer(self):
        if self.sipCallState == SipCallState.RINGING or self.sipCallState == SipCallState.INCOMMING_CALL:
            if self.current_call is not None and not self.on_call:
                self.isWaitingToAnswer = False
                self.logger.info("will answer call")
                self.current_call.answer(200) 
                self.on_call = True
        self.requestToAnswerCall = False

    def hangup(self):
        self.stopTone()
        if self.current_call is not None:
            try:
                self.current_call.hangup()
            except:
                self.logger.error("Exception hanging up call...", exc_info=1 )
            self.current_call = None
        self.requestToHangupCall = False

    def call(self):
        self.logger.debug("current_call:{}, on_call:{}, self.dstSipUri:{}".format(self.current_call,self.on_call,self.dstSipUri))
        if self.current_call is None and not self.on_call and self.dstSipUri is not None:
            self.current_call = self.acc.make_call(self.dstSipUri, cb=MyCallCallback(sipClient=self))
            self.on_call = True
        self.requestToMakeCall = False

    def startTone(self, tone, loop=True):
        try:
            if os.path.isfile(tone):
                self.wavPlayerId = self.lib.create_player(tone,loop)
                self.wavPlayerSlot = self.lib.player_get_slot(self.wavPlayerId)
                # (wav file player) slot x => slot 0 (sounde device)
                self.lib.conf_connect(self.wavPlayerSlot, 0)
                self.logger.info("Connect wavPlayer tone {} slot {} to conference".format(tone,self.wavPlayerSlot))
            else:
                self.logger.error("tone audio file not found")
        except:
            self.logger.error("Exception startTone", exc_info=1 )

    def stopTone(self):
        try:
            if self.wavPlayerSlot is not None:
                self.lib.conf_disconnect(self.wavPlayerSlot, 0)
                self.logger.info("Remove wavPlayer tone slot {} from conference".format(self.wavPlayerSlot))
            if self.wavPlayerId is not None:
                self.lib.player_destroy(self.wavPlayerId)
                self.wavPlayerId = None
                self.wavPlayerSlot = None
        except:
            self.logger.error("Exception stopTone", exc_info=1 )

    def checkBusyTone(self):
        if ( time.monotonic() - self.busyToneStartTime ) >= self.BUSY_TONE_PLAY_TIME:
            self.logger.info("stop busy tone timeout of {} seconds reached".format(self.BUSY_TONE_PLAY_TIME))
            self.disconnectCallState()
            self.requestCheckBusyTone = False

    def checkCallTimeout(self):
        if ( time.monotonic() - self.callStartTime ) >= self.callTimeout:
            self.logger.info("call timeout of {} seconds reached, will hangup call".format(self.callTimeout))
            self.hangup()
            self.requestCheckCallTimeout = False
    
    """
    there's no pjsua api to check if we receive packets so we check the audio rx level
    if it does not change during the stream timeout time we'll hangup
    """
    def checkStreamTimeout(self):
        try:
            if self.current_call is not None:
                if self.current_call.is_valid():
                    if self.current_call.info().state == pj.CallState.CONFIRMED:
                        currTime = time.monotonic()
                        current_call_slot = self.current_call.info().conf_slot
                        if current_call_slot is not None:
                            if self.current_call.info().media_state == pj.MediaState.ACTIVE:
                                tx_level, rx_level = self.lib.conf_get_signal_level(current_call_slot)
                                if rx_level != self.rxLastLevel:
                                    self.streamStartTime = currTime
                                    self.rxLastLevel = rx_level
                        
                    if ( currTime - self.streamStartTime ) >= self.streamTimeout:
                        self.logger.info("stream timeout of {} seconds reached, will hungup call".format(self.streamTimeout))
                        self.hangup()
                        self.requestCheckStreamTimeout = False
        except:
            self.logger.debug("Invalid rx signal checking - maybe the call was disconnected",exc_info=1)
            pass

    def connectCallState(self):
        self.stopTone()
        try:
            if self.call_slot:
                if self.beepOnAnswer and self.incommingCallReceived:
                    self.logger.debug("connectCallState beepOnAnswer")
                    self.logger.debug("Starting beep")
                    self.startTone(self.beepTone,False)
                    time.sleep(1)
                    self.logger.debug("Stopping beep")
                    self.stopTone()
                    self.incommingCallReceived = False
                
                if self.current_call.is_valid():
                    self.lib.conf_set_tx_level(self.call_slot,self.inputGain)
                    self.logger.info("Connect Audio Call Slot :\"{}\" ".format(self.call_slot))
                    self.lib.conf_connect(self.call_slot, 0)
                    self.lib.conf_connect(0, self.call_slot)
                    self.logger.info("Audio Connection Complete")
                    self.sipCallState = SipCallState.CONNECTED
                    if self.halfDuplex:
                        self.startTimeHalfDuplex = time.monotonic()
                        self.requestCheckHalfDuplex = True
                    if self.callTimeout != 0:
                        self.callStartTime = time.monotonic()
                        self.logger.info("Check for Call Timeout")
                        self.requestCheckCallTimeout = True
                    if self.streamTimeout != 0:
                        self.streamStartTime = time.monotonic()
                        self.rxLastLevel = 0
                        self.logger.info("Check for Stream Timeout")
                        self.requestCheckStreamTimeout = True
        except:
            self.logger.error("connect Call exception",exc_info=1)

    def disconnectCallState(self):
        self.logger.info("Current call is {}".format(self.current_call))
        self.on_call = False
        self.current_call = None
        self.alive = False
        self.logger.info("Disconnect Audio Call Slot :\"{}\" ".format(self.call_slot))
        self.sipCallState = SipCallState.IDLE
        self.isWaitingToAnswer = False
        AudioSource.sendSourceStateUpdate(self,self.state,self.alive)
        self.call_slot  = None
        self.stopTone()
        self.requestCheckCallTimeout = False
        self.requestCheckStreamTimeout = False
        self.incommingCallReceived = False
        self.requestCheckBusyTone = False
        self.requestCheckHalfDuplex = False
        self.requestToDisconnect = False

    def processHalfDuplex(self):	    
        if self.current_call is not None:
            try:
                if self.current_call.is_valid():
                    if self.current_call.info().state == pj.CallState.CONFIRMED:
                        current_call_slot = self.current_call.info().conf_slot
                        if current_call_slot is not None:
                            current_media_state = self.current_call.info().media_state 
                            if current_media_state == pj.MediaState.ACTIVE: 
                                tx_level, rx_level = self.lib.conf_get_signal_level(current_call_slot)
                                curr_dur = time.monotonic() - self.startTimeHalfDuplex
                                rx_diff = rx_level - self.hdxLevel
                                if rx_diff > 0:
                                    self.startTimeHalfDuplex = time.monotonic()
                                    if self.halfDuplexMode == HalfDuplexState.TALKING:
                                        self.logger.debug("Half Duplex talk mode LISTENING ...")
                                        self.lib.conf_set_tx_level(current_call_slot,0)
                                        self.halfDuplexMode = HalfDuplexState.LISTENING
                                else:
                                    if curr_dur > self.hdxTimeout and self.halfDuplexMode == HalfDuplexState.LISTENING:
                                        self.logger.debug("Half Duplex talk mode: TALKING ...")
                                        self.lib.conf_set_tx_level(current_call_slot,self.inputGain)
                                        self.halfDuplexMode = HalfDuplexState.TALKING
            except:
                self.logger.error("processHalfDuplex: conf_get_signal_level invalid call slot id",exc_info=1)

    def restartRegistration(self):
        """
        used to re-registration upon receiving 401 Unauthorized or 403 Forbidden
        """
        if ( time.monotonic() - self.reqRegStartTime ) >= self.regTimeout:
            self.acc.set_registration(True)
            self.requestToRegister = False

        
    def onAccountEvent(self,eventType,eventParams=None):
        self.logger.info("ON ACCOUNT EVENT - {}".format(eventType))
        if eventType == SipAccountEvent.REGISTERED: 
            self.logger.info("SIP SERVER REGISTERED")
            self.registered = True

        if eventType == SipAccountEvent.UNREGISTERED:
            self.logger.info("SIP SERVER UNREGISTERED")
            self.registered = False
            self.call_slot  = None
            if self.sipRegStatus in [401,403]:
                self.logger.info(f"Received {self.sipRegStatus} from SIP server... will try to register in 30 seconds")
                self.reqRegStartTime = time.monotonic()
                self.requestToRegister = True

        if eventType == SipAccountEvent.INCOMMING_CALL:
            self.logger.info("INCOMMING_CALL")
            if self.on_call:
                self.logger.info("Busy - Already on call")
                eventParams.answer(486, "Busy")
                return

            self.sipCallState = SipCallState.INCOMMING_CALL
            self.incommingCallReceived = True
            self.current_call = eventParams
            call_cb = MyCallCallback(self)
            self.current_call.set_callback(call_cb)
            self.current_call.answer(180)
            
            if self.autoAnswer:
                if self.timeToAnswer == 0:
                    self.current_call.answer(200)
                    self.on_call = True
                else:
                    if not self.isWaitingToAnswer:
                        self.startTimeAnswer = time.monotonic()
                        self.isWaitingToAnswer = True
                        self.logger.debug("isWaintingToAnswer:{}".format(self.isWaitingToAnswer))

    def onCallEvent(self,eventType,eventParams=None):
        if eventType == SipCallEvent.CALL_DISCONNECT:
            self.requestToDisconnect = True
        
        if eventType == SipCallEvent.MEDIA_ACTIVE:
            self.call_slot = eventParams["slot"]
            self.logger.info( "OnCallEvent MEDIA_ACTIVE call_slot:\"{}\" ".format(self.call_slot))
        
        if eventType == SipCallEvent.MEDIA_INACTIVE:
            self.logger.info("Media Inactive")
            self.alive = False
            AudioSource.sendSourceStateUpdate(self,self.state,self.alive)

        if eventType == SipCallEvent.CALL_RINGING:
            if self.incommingCallReceived:
                self.logger.info("Ringing after receiving an incoming call")
                self.sipCallState = SipCallState.RINGING
            else:
                self.logger.info("Ringing after making a call")
                self.alive = True
                AudioSource.sendSourceStateUpdate(self, self.state, self.alive)
                self.sipCallState = SipCallState.CALLING_RINGING

        if eventType == SipCallEvent.CALL_CALLING:
            self.logger.info("Calling ...")
            self.startTone(self.waitTone)
            self.sipCallState = SipCallState.CALLING
            self.alive = True
            AudioSource.sendSourceStateUpdate(self, self.state, self.alive)

        if eventType == SipCallEvent.CALL_CONNECTING:
            self.logger.info("Call Connecting")
            if not self.incommingCallReceived and not self.alive: # corner case where it bypassed the Ringing state
                self.alive = True
                AudioSource.sendSourceStateUpdate(self,self.state,self.alive)

        if eventType == SipCallEvent.CALL_CONFIRMED:
            self.logger.info("Call Confirmed")
            if self.incommingCallReceived and not self.alive:
                self.alive = True
                AudioSource.sendSourceStateUpdate(self,self.state,self.alive)
            self.requestToConnectCall = True

        if eventType == SipCallEvent.CALL_BUSY:
            self.logger.info("Call Busy ...")
            self.stopTone()
            self.startTone(self.busyTone)
            self.busyToneStartTime = time.monotonic()
            self.requestCheckBusyTone = True

    def reportStatus(self):
        sourceDict               = {}
        sourceDict["curState"]      = "playing" if self.state==SourceState.PLAYING else ("stopped" if self.state==SourceState.STOPPED else "monitoring")
        sourceDict["state"]      = "alive" if self.alive else "dead"
        sourceDict["callState"]  = self.sipCallState
        sourceDict["counter"]    = self.counter
        sourceDict["currentCall"]= self.currentCall_id if self.on_call else None
        if self.sip_mode != "peer_to_peer":
            sourceDict["registered"] = "registered" if self.registered else "unregistered"
            sourceDict["sipResponseCode"] = self.reportSipResponseCode
        else:
            sourceDict["registered"] = None
        if self.currentCall_id is not None:
            if "@" in self.currentCall_id:
                sourceDict["transSecure"] = self.reportTransSecure
            else:
                sourceDict["transSecure"] = "No"

        return sourceDict

    def reportSipCallState(self):
        return self.sipCallState        

    def disconnectPJSUA(self):	
        if self.acc:
            self.acc.delete()
        if self.lib:
            self.lib.destroy()

    def disconnect(self):
        self.requestToDisconnectPJSUA = True
        
    def __del__(self):
        self.disconnect()



# Callback to receive events from account
class MyAccountCallback(pj.AccountCallback):
    
    def __init__(self, account=None , sipClient=None ):
        pj.AccountCallback.__init__(self, account)
        self.account = account
        self.sipClient    = sipClient
        self.logger = logging.getLogger('sip')

    def on_reg_state(self):
        self.logger.info("SIP Client reg_active:{} reg_status:{} reg_reason:{}".format(self.account.info().reg_active,self.account.info().reg_status, self.account.info().reg_reason))
        self.sipClient.reportSipResponseCode = "{} {}".format(self.account.info().reg_status, self.account.info().reg_reason)
        self.sipClient.sipRegStatus = self.account.info().reg_status
        if self.account.info().reg_active and self.account.info().reg_status == 200:
            self.sipClient.onAccountEvent(SipAccountEvent.REGISTERED)
        else:
            self.sipClient.onAccountEvent(SipAccountEvent.UNREGISTERED)

    # Notification on incoming call
    def on_incoming_call(self, call):
        self.logger.info("Incoming call from {}. Received in account: {}".format(call.info().remote_uri,self.account))
        self.sipClient.currentCall_id = call.info().remote_uri
        self.sipClient.onAccountEvent(SipAccountEvent.INCOMMING_CALL, call)


# Callback to receive events from Call
class MyCallCallback(pj.CallCallback):
    def __init__(self, sipClient=None):
        pj.CallCallback.__init__(self, sipClient.current_call)
        self.sipClient  = sipClient
        self.logger = logging.getLogger('sip')

    # Notification when call state has changed
    def on_state(self):
        self.logger.info("Call with {} is {} last code ={} ( {} )".format(self.call.info().remote_uri,self.call.info().state_text, self.call.info().last_code,self.call.info().last_reason))
        self.sipClient.currentCall_id = self.call.info().remote_uri

        if self.call.info().state == pj.CallState.EARLY and self.call.info().last_code == 180:
            self.sipClient.onCallEvent(SipCallEvent.CALL_RINGING)
        
        if self.call.info().state == pj.CallState.CALLING:
            self.sipClient.onCallEvent(SipCallEvent.CALL_CALLING)

        if self.call.info().state == pj.CallState.CONNECTING:
            self.sipClient.onCallEvent(SipCallEvent.CALL_CONNECTING)

        if self.call.info().state == pj.CallState.CONFIRMED:
            self.sipClient.onCallEvent(SipCallEvent.CALL_CONFIRMED)

        if self.call.info().last_code == 486:
            self.sipClient.onCallEvent(SipCallEvent.CALL_BUSY)

        if self.call.info().state == pj.CallState.DISCONNECTED and self.call.info().last_code != 486:
            self.sipClient.onCallEvent(SipCallEvent.CALL_DISCONNECT)

    # Notification when call's media state has changed.
    def on_media_state(self):
        if self.call.info().media_state == pj.MediaState.ACTIVE:
            # Connect the call to sound device
            call_slot = self.call.info().conf_slot
            self.logger.info("Media is now active -- call_slot:{}".format(call_slot))
            self.sipClient.onCallEvent(SipCallEvent.MEDIA_ACTIVE, {"slot" : call_slot} )

        else:
            self.logger.info("Media is inactive")
            self.sipClient.onCallEvent(SipCallEvent.MEDIA_INACTIVE)
