/**
 * \file splayerapi-5.h
 * \brief The public API of the SPLAYER library.
 * \copyright Copyright Soundtrack Your Brand AB 2014.
 *
 * SOUNDTRACK YOUR BRAND SWEDEN AB - CONFIDENTIAL
 * __________________
 *
 *  [2013] - SOUNDTRACK YOUR BRAND SWEDEN AB
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of SOUNDTRACK YOUR BRAND SWEDEN AB and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to SOUNDTRACK YOUR BRAND SWEDEN AB
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret and copyright law.
 * Dissemination of this information or reproduction, sub-licensing or
 * modification of this material is strictly forbidden unless prior written
 * permission is obtained from SOUNDTRACK YOUR BRAND SWEDEN AB.
 * Violations of these rights will result in legal actions.
 *
 * https://www.soundtrackyourbrand.com/legal/sdk-terms-of-use
 */

#ifndef _SPLAYER_H
#define _SPLAYER_H

#if __GNUC__ >= 4
#define SPLAYER_PUBLIC __attribute__((__visibility__("default")))
#elif defined(_MSC_VER)
#ifdef SPLAYER_EXPORTS
#define SPLAYER_PUBLIC __declspec(dllexport)
#else
#define SPLAYER_PUBLIC __declspec(dllimport)
#endif
#else
#define SPLAYER_PUBLIC
#endif

#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief Struct containing major and minor version number of current SDK
 */
typedef struct {
  int major;
  int minor;
} sdk_version_t;

/**
 * \brief SPLAYER SDK Version must match the version in library.
 */
static const sdk_version_t SPLAYER_SDK_VERSION = {5, 0};

/**
 * \brief This is the single entry point for SPLAYER API exposing the rest of the API calls
 * through a struct. This way new functions call be added in a backward compatible way
 * with minimal load time.
 */
#define SPLAYER_API SPLAYER_API_87GHV3
/**
 * \brief Symbol to do dlsym on if using shared library api.
 *        eg const struct splayer_api *api = dlsym(dl_handle, SPLAYER_API_SYMBOL_3_0);
 */
#define SPLAYER_API_SYMBOL "SPLAYER_API_87GHV3"

/**
 * These symbols are defined in the library, and shared among all the sdk:s
 */
SPLAYER_PUBLIC extern const char* SPLAYER_VERSION;
SPLAYER_PUBLIC extern const char* SPLAYER_BUILD;
SPLAYER_PUBLIC extern const char* SPLAYER_REV;
SPLAYER_PUBLIC extern const char* SPLAYER_PLATFORM_NAME;

/**
 * \brief Opaque type to SPLAYER API context.
 */
typedef struct splayer splayer_t;

/**
 * \brief 0 = SUCCESS, non-zero = FAILED
 */
typedef int err_t;

/**
 * \brief Config used to create and initialize the player
 */
typedef struct {
  /**
   * \brief SDK version must match the version in library.
   * \note This field is required to be set.
   */
  sdk_version_t sdk_version;

  /**
   * \brief Pointer to users implementation of the callbacks.
   * \note Needs to point to valid memory during the entire lifetime of the application.
   */
  struct splayer_audio_api* audio_api_callbacks;

  /**
   * \brief DEPRECATED: This field is no longer required.
   */
  int hardware_bandwidth_limitation_kbps;

  /**
   * \brief Path where to store your cache directory. Default is workdir/cache.
   * \note You can destroy the buffer holding the path after calling create.
   */
  const char* diskcache_dir;

  /**
   * \brief Maximum size of disk cache.
   * A value > 0 will fix the max size of the disk cache. This will disable the possibility to change the size from
   * Soundtrack. A value 0 will result in a default value to be set (currently 5 GB, but subject to change). This will
   * enable the possibility to change the max size from Soundtrack. \note You can destroy the buffer holding the path
   * after calling create.
   */
  int diskcache_max_mb;

  /**
   * \brief Minimum free storage on the disk holding the disc cache.
   */
  int diskcache_remain_mb;

  /**
   * \brief This is the number of channels of your soundcard. Eg 1 (mono) or 2 (stereo)
   * DSP chain will provide you with either mono or stereo sound. If the source is mono and you chose stereo for
   * output_sample_channels the audio will be re-sampled to stereo. And if you then choose mono for your soundcard it
   * will be down-sampled back to mono.
   * \note To avoid unnecessary re-sampling from mono to stereo or vice versa, choose the same number of channels
   * for both output_sample_channels rate AND your soundcard.
   * Eg. Whatever channels you provide to snd_pcm_set_params() should be the same as output_sample_channels.
   */
  int output_sample_channels;

  /**
   * \brief This is the sample rate of your soundcard. Eg 44100 Hz or 48000Hz.
   * DSP chain will provide you with these sample rates. If the sound file played has a sample rate of 48 kHz
   * and the output_sample_rate is set to 44.1 kHz, then the audio will be downsampled. If your soundcard (ALSA)
   * is set up at at sample rate of 48 kHz, then it will be up-sampled.
   *
   * \note To avoid unnecessary re-sampling, choose the same sample rate for both output_sample rate AND your soundcard.
   * Eg. Whatever sampler rate you provide to snd_pcm_set_params() should be the same as output_sample_rate
   */
  int output_sample_rate;

  /**
   * \brief DEPRECATED: This field is no longer required.
   */
  const char* vendor_hardware_id;

  /**
   * \brief Name of the device running splayer.
   * \note This field is required to be set.
   */
  const char* vendor_device_name;

  /**
   * \brief DEPRECATED: This field is no longer required.
   */
  const char* vendor_secret;

  /**
   * \brief App version
   * \note Your current Application version, this is used for logging and troubleshooting.
   *       format should be [1-9][0-9]*\.[0-9]+ (eg. 1.0, 2.3, 10.20 etc)
   * \note This field is required to be set.
   */
  const char* app_version;

  /**
   * \brief This setting allows you to disable core dumps if set to true.
   */
  bool disable_core_dumps;

  /**
   * \brief DEPRECATED: This field is no longer required.
   */
  bool use_interactive_pairing;

  /**
   * \brief This setting allows you to disable the on-disk caching of played audio files.
   * \note Enabling this setting increases network requests and data downloads, but is only
   *       recommended for platforms with limited storage.
   *       Note that it disables offline audio playback and renders diskcache_max_mb and
   *       diskcache_remain_mb settings irrelevant.
   */
  bool disable_audio_caching;

  /**
   * \brief This callback function allows you to add custom logic when loading the token.
   * \note A custom callback function which overrides default token loading logic.
   *       If an invalid token is returned, pairing will fail.
   */
  void (*load_token_callback)(char* buf, size_t buflen);

  /**
   * \brief This callback function allows you to add custom logic when saving the token.
   * \note A custom callback function which overrides default token saving logic.
   *       Store this token to load it using load_token_callback.
   */
  void (*save_token_callback)(const char*);

} splayer_config_t;

/**
 * \brief Sets the quit flag in the session.
 */
typedef enum {
  /**
   * \brief Wait for all tasks and threads to finish and quit.
   */
  SPLAYER_EXIT_NORMAL = 0,

  /**
   * \brief Restart after finishing playing the current song.
   */
  SPLAYER_EXIT_RESTART = 1,

  /**
   * \brief Quit after finishing playing the current song.
   */
  SPLAYER_EXIT_END_OF_SONG = 2,

  /**
   * \brief Restart and upgrade after finishing playing the current song.
   */
  SPLAYER_EXIT_UPGRADE = 3,

  /**
   * \brief Quit now.
   */
  SPLAYER_EXIT_HARD = 4
} splayer_exit_t;

/**
 * \brief This is the struct declaring all API calls for SPLAYER API. The simplest use case
 * is to first create a splayer_t context. Don't forget to create a splayer_audio_api struct, with
 * the callback code and set config.audio_api_callbacks variable. splayer_audio_api struct needs
 * to be valid memory during the entire lifetime of the application. Then call loop_iteration() or
 * should_exit() periodically to play a audio and finally call free() when you are done.
 */
struct splayer_api {
  /**
   * \brief SPLAYER SDK version must match the version in library.
   */
  sdk_version_t splayer_sdk_version;

  /**
   * \brief Creates a SPLAYER API context, that you need to store somewhere (eg. on the stack). All values are copied
   * from config, so config parameter could be stored on the stack or freed after calling create.
   * \note config.audio_api_callbacks needs to be valid memory during the lifetime of the application
   * \param[in] config Configuration for the SPLAYER API.
   * \param[out] SPLAYER API context, that you will need to call the rest of the API functions and current and future
   * extensions. SPLAYER_CONTROLS_API is such an extension
   * \return Returns 0 if success otherwise non-zero.
   */
  err_t (*create)(const splayer_config_t config, splayer_t** splayer);

  /**
   * \brief Frees the SPLAYER API context.
   * \param[in] splayer SPLAYER API context.
   */
  void (*free)(splayer_t* splayer);

  /**
   * \brief Call this to retrieve the librarys current player version.
   * \param[in] splayer SPLAYER API context.
   * \param[out] version A buffer where the version string will be placed. Recommended size is at least 128 bytes.
   * \param[in] version_size The allocated buffer's size. Recommended size is at least 128 bytes.
   */
  void (*get_current_version)(splayer_t* splayer, char* version, int version_size);

  /**
   * \brief Cranks the syb engine and plays sound by calling the callbacks. Keep calling loop_iteration() indefinitely
   * unless it returns zero (0)
   * This method will block and can safely be called in a loop.
   *
   * \note This function might block, so it is not suitable for all applications. If not suitable, consider using
   * `should_exit` instead.
   *
   * \param[in] splayer SPLAYER API context.
   * \return Returns 0 when something went wrong and quit the application. Keep calling if non-zero.
   */
  int (*loop_iteration)(splayer_t* splayer);

  /**
   * \brief Call this if you want the player to quit gracefully, e.g on the next song.
   * \param[in] splayer SPLAYER API context.
   * \param[in] exit_method For more info checkout splayer_exit_t.
   */
  void (*request_exit)(splayer_t* splayer, splayer_exit_t exit_method);

  /**
   * \brief Call this to get a pointer to handle to the Troubles API
   * \return Returns a pointer to the Troubles API
   */
  const struct splayer_troubles_api* (*get_troubles_api)();
  /**
   * \brief Call this to get a pointer to handle to the Controls API
   * \return Returns a pointer to the Controls API
   */
  const struct splayer_controls_api* (*get_controls_api)();
  /**
   * \brief Call this to get a pointer to handle to the Metadata API
   * \return Returns a pointer to the Metadata API
   */
  const struct splayer_metadata_api* (*get_metadata_api)();
  /**
   * \brief Call this to get a pointer to handle to the Auth API
   * \return Returns a pointer to the Auth API
   */
  const struct splayer_auth_api* (*get_auth_api)();

  /**
   * \brief Check if the application is to be restarted.
   * This function is an alternative to calling loop_iteration() in a loop, which is suitable for applications that
   * can't call a blocking function in a loop.
   * \param[in] splayer SPLAYER API context.
   * \return Returns 1 if the application is to be restarted, otherwise 0.
   */
  int (*should_exit)(splayer_t*);
};

/**
 * \brief This is the structure holding function pointers to the SPLAYER API
 * \note You should use the SPLAYER_API define to get this struct in a backwards compatible way
 */
SPLAYER_PUBLIC extern const struct splayer_api SPLAYER_API_87GHV3;

#ifdef __cplusplus
} // extern C
#endif

#endif // _SPLAYER_H/
