/**
 * \file splayer_audio_api-3.h
 * \brief The public API Audio of the SPLAYER library.
 * \copyright Copyright Soundtrack Your Brand AB 2014.
 *
 * SOUNDTRACK YOUR BRAND SWEDEN AB - CONFIDENTIAL
 * __________________
 *
 *  [2013] - SOUNDTRACK YOUR BRAND SWEDEN AB
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of SOUNDTRACK YOUR BRAND SWEDEN AB and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to SOUNDTRACK YOUR BRAND SWEDEN AB
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret and copyright law.
 * Dissemination of this information or reproduction, sub-licensing or
 * modification of this material is strictly forbidden unless prior written
 * permission is obtained from SOUNDTRACK YOUR BRAND SWEDEN AB.
 * Violations of these rights will result in legal actions.
 *
 * https://www.soundtrackyourbrand.com/legal/sdk-terms-of-use
 */

#ifndef _SPLAYER_AUDIO_API_H
#define _SPLAYER_AUDIO_API_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stddef.h>
#include <stdint.h>

/**
 * \brief Callback Audio API where your (the users) platform/hardware specific code exists. Write your callback
 * functions and make a splayer_audio_api struct out of them. The struct needs to be valid memory during
 * the entire runtime of the application. Assign it to config.audio_api_callbacks, before you call loop_iteration.
 */
typedef struct splayer_audio_api {

  /**
   * \brief Initialization callback.
   * This function is called when the audio pipeline is setup.
   * An init and shutdown might be run in series if splayer detects that
   * audio data is not delivered in a 60 second window, to attempt
   * to restore the operation.
   */
  void (*audio_init)(struct splayer_audio_api* ctx, const int output_sample_channels, const int output_sample_rate);

  /**
   * \brief Shutdown callback.
   * Will be called when the audio pipeline is shutting down.
   */
  void (*audio_shutdown)(struct splayer_audio_api* ctx);

  /**
   * \brief Flush the buffer.
   * Will be called when a sudden track change is required and all
   * the buffers should be emptied (including harware buffers).
   */
  void (*audio_flush)(struct splayer_audio_api* ctx);

  /**
   * \brief Main audio callback.
   * Audio is delivered by this callback in 16 bit signed integer pcm samples.
   * The *sample parameter points to memory, where sample_count samples might be read.
   * The number of samples consumed from this buffer, is returned as the return value.
   * In the memory pointed to by *samples_buffered, you have to set the size in bytes of your buffer
   * all the way to the hardware, so we can calculate how big delay there is on the audio.
   *
   * \note audio_data() function can NOT be blocked, as its the same thread responsible for decoding the
   * playback content. If a blocking operation is required, a thread with a circular buffer
   * is recommended.
   */
  size_t (*audio_data)(struct splayer_audio_api* ctx, const int16_t* samples, size_t sample_count,
                       uint32_t* samples_buffered);

  /**
   * \brief Pause output.
   * This will be called if user ask to pause audio output.
   * You should not flush buffers on unpause, just continue delivering the audio.
   */
  void (*audio_pause)(struct splayer_audio_api* ctx, int pause);

  /**
   * \brief Adjust volume.
   * vol is an integer between 0 and 100, inclusive.
   */
  void (*audio_volume)(struct splayer_audio_api* ctx, int vol);
} splayer_audio_api_t;

#ifdef __cplusplus
} // extern C
#endif

#endif // _SPLAYER_AUDIO_API_H/
