#!/bin/sh

# Barix I2C GPIO Expander utilities.
# Copyright (C) 2018 Barix AG (http://www.barix.com). All rights reserved.
#
# Dependencies from 'qiba-exstreamerl.sh' main script:
#  Includes
#   + 'qiba-log.sh': Logging facilities.
#   + 'qiba-gpio.sh': GPIO utilities.
#  Variables
#   + 'TEST_HOST: Testing on host.

# The sys path root.
# Allowed to be specified externally, e.g., for testing.
[ -z "${SYSPATH}" ] && SYSPATH=""

# The log messages prefix.
# Allowed to be specified externally.
[ -z "${LOG_PFX}" ] && LOG_PFX="qiba-i2c-gpio"

# The sys path for I2C devices.
readonly SYSPATH_I2C="${SYSPATH}/sys/bus/i2c/devices"

# I2C: Get I2C device ID from I2C bus and address.
#
# Parameters:
#  $1: The device's I2C bus number (dec, hex, octal accepted).
#  $2: The device's I2C address (dec, hex, octal accepted).
#  $3: [opt] Variable to store the output.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
#  <echo>|<outp>: The I2C device ID, in the 'B-AAAA' format, where 'B' is the
#   I2C bus number and 'AAAA' is the I2C hex address. E.g.:
#    + bus='0',   addr='0x20' ==> id = '0-0020'.
#    + bus='0',   addr='32'   ==> id = '0-0020'.
#    + bus='0x1', addr='0x21' ==> id = '1-0021'.
_i2c_dev_id_get() {
	local bus="${1}"
	local addr="${2}"
	local outp="${3}"

	local _dev_elems="${bus} ${addr}"
	local _dev_id

	local ret=0
	local f="${FUNCNAME[0]}"

	if [ -z "${bus}" ] || [ -z "${addr}" ]; then
		log_err "${f}" "Missing I2C bus/addr parameters"
		return 1
	fi

	# On target, 'awk' doesn't know about 'strtonum()' but deals well with
	# hex/dec/octal numbers. On the other hand, the host does know and needs
	# 'strtonum()'.
	if [ "${TEST_HOST}" = "1" ]; then
		_dev_id=$(awk '{printf("%d-%04x", strtonum($1), strtonum($2))}' 2>&1 \
			<<< "${_dev_elems}") || ret=$?
	else
		_dev_id=$(awk '{printf("%d-%04x", $1, $2)}' 2>&1 \
			<<< "${_dev_elems}") || ret=$?
	fi
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get I2C device ID (${ret}):"
		log_err "${f}" "${_dev_id}"
		return ${ret}
	fi

	# If there is a variable to store the output, use it. Otherwise just
	# return the output through echoing.
	[ -n "${outp}" ] && eval ${outp}="'${_dev_id}'" || echo "${_dev_id}"

	return 0
}

# I2C GPIO: Checks if GPIO base/num is inside valid GPIO range.
#
# Parameters:
#  $1: The GPIO zero based offset.
#  $2: The number of GPIOs.
#  $3: The valid GPIO range base.
#  $3: The valid number of GPIOs.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
_i2c_gpio_range_check() {
	local base="${1}"
	local num="${2}"
	local gpio_base="${3}"
	local gpio_num="${4}"

	local start range

	local ret=0
	local f="${FUNCNAME[0]}"

	if [ -z "${base}" ]; then
		log_err "${f}" "Missing base GPIO number"
		return 1
	fi
	if [ -z "${num}" ]; then
		num="1"
	fi
	if [ "${base}" -lt "0" ]; then
		log_err "${f}" "Invalid base GPIO '${base}'"
		return 1
	fi
	start=$((${gpio_base} + ${base}))
	if [ ${start} -ge $((${gpio_base} + ${gpio_num})) ]; then
		log_err "${f}" "GPIO '${start}' out of range"
		return 1
	fi
	if [ $((${start} + ${num})) -gt $((${gpio_base} + ${gpio_num})) ]; then
		range="${start}..$((${start} + ${num} - 1)) (${num})"
		log_err "${f}" "GPIOs '${range}' out of range"
		return 1
	fi

	return 0
}

# I2C GPIO: Get I2C GPIO Expander data (base number and number of GPIOs).
#
# Parameters:
#  $1: The expander's I2C bus number (dec, hex, octal accepted).
#  $2: The expander's I2C address (dec, hex, octal accepted).
#  $3: [opt] Variable to store the GPIO base number output.
#  $4: [opt] Variable to store the number of GPIOs output.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
#  <outp1>: The GPIO base number, placed in var specified by param $1.
#  <outp2>: The Number of GPIOs, placed in var specified by param $2.
#  <echo>: The GPIO data, in the form of '<base> <ngpio>', where <base> is the
#   base GPIO number and <ngpio> is the number of GPIOs, e.g. '400 16'. Echo
#   output only sent if none of params $2 and $3 specified.
i2c_gpio_data_get() {
	local bus="${1}"
	local addr="${2}"
	local outp1="${3}"
	local outp2="${4}"

	local i2c_dev sys_path i2c_gpio_base i2c_gpio_num

	local ret=0
	local f="${FUNCNAME[0]}"

	# Get the I2C device ID.
	_i2c_dev_id_get "${bus}" "${addr}" i2c_dev || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get I2C device ID (${ret}):"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	# Get the GPIO data.
	sys_path="${SYSPATH_I2C}/${i2c_dev}"
	gpio_data_get "${sys_path}" i2c_gpio_base i2c_gpio_num || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get GPIO data (${ret})"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	# If there is a variable to store the output, use it. Otherwise just
	# return the output through echoing.
	if [ -n "${outp1}" ]; then
		eval ${outp1}="'${i2c_gpio_base}'"
		[ -n "${outp2}" ] && eval ${outp2}="'${i2c_gpio_num}'"
		return 0
	fi
	echo "${base} ${ngpio}"
	return 0
}

# I2C GPIO: Export the GPIOs from an I2C GPIO Expander.
#
# Parameters:
#  $1: The expander's I2C bus number (dec, hex, octal accepted).
#  $2: The expander's I2C address (dec, hex, octal accepted).
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
i2c_gpio_export() {
	local bus="${1}"
	local addr="${2}"

	local base num

	local res
	local ret=0
	local f="${FUNCNAME[0]}"

	# Get the I2C GPIO data.
	i2c_gpio_data_get "${bus}" "${addr}" base num || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get I2C GPIO data (${ret})"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	# Export all GPIOs.
	gpio_export "${base}" "${num}" || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't export GPIOs (${ret})"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	return 0
}

# I2C GPIO: Unexport the GPIOs from an I2C GPIO Expander.
#
# Parameters:
#  $1: The expander's I2C bus number (dec, hex, octal accepted).
#  $2: The expander's I2C address (dec, hex, octal accepted).
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
i2c_gpio_unexport() {
	local bus="${1}"
	local addr="${2}"

	local base num

	local res
	local ret=0
	local f="${FUNCNAME[0]}"

	# Get the I2C GPIO data.
	i2c_gpio_data_get "${bus}" "${addr}" base num || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get I2C GPIO data (${ret})"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	# Export all GPIOs.
	gpio_unexport "${base}" "${num}" || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't unexport GPIOs (${ret})"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	return 0
}

# I2C GPIO: Set the direction of I2C GPIO Expander GPIOs.
#
# Parameters:
#  $1: The expander's I2C bus number (dec, hex, octal accepted).
#  $2: The expander's I2C address (dec, hex, octal accepted).
#  $3: The GPIO direction, 'in' or 'out'.
#  $4: The expander's GPIO base number (starting at zero).
#  $5: [opt] The number of GPIOs. If not specified, 1 is assumed.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
i2c_gpio_dir_set() {
	local bus="${1}"
	local addr="${2}"
	local dir="${3}"
	local base="${4}"
	local num="${5}"

	local gpio_base gpio_num start

	local ret=0
	local f="${FUNCNAME[0]}"

	# Get the I2C GPIO data.
	i2c_gpio_data_get "${bus}" "${addr}" gpio_base gpio_num || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get I2C GPIO data (${ret})"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	# Validate the GPIOs base/num against valid gpio_base/gpio_num.
	_i2c_gpio_range_check "${base}" "${num}" "${gpio_base}" "${gpio_num}" \
		|| ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" \
			"Invalid GPIOs '${base}..$((${base} + ${num} - 1)) (${num})'"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi
	start=$((${gpio_base} + ${base}))

	gpio_dir_set "${dir}" "${start}" "${num}" || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Could not set direction in all '${num}' GPIOs (${ret})"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	return 0
}

# I2C GPIO: Get the direction of I2C GPIO Expander GPIO.
#
# Parameters:
#  $1: The expander's I2C bus number (dec, hex, octal accepted).
#  $2: The expander's I2C address (dec, hex, octal accepted).
#  $3: The expander's GPIO number (starting at zero).
#  $4: [opt] Variable to store the output.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
#  <echo>|<outp>: The GPIO direction, 'in' or 'out'.
i2c_gpio_dir_get() {
	local bus="${1}"
	local addr="${2}"
	local num="${3}"
	local outp="${4}"

	local gpio_base gpio_num start dir

	local ret=0
	local f="${FUNCNAME[0]}"

	# Get the I2C GPIO data.
	i2c_gpio_data_get "${bus}" "${addr}" gpio_base gpio_num || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get I2C GPIO data (${ret})"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	# Validate the GPIOs base/num against valid gpio_base/gpio_num.
	_i2c_gpio_range_check "${num}" "1" "${gpio_base}" "${gpio_num}" \
		|| ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Invalid GPIO '${num}'"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi
	start=$((${gpio_base} + ${num}))

	gpio_dir_get "${start}" dir || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get GPIO '${num}' direction (${ret})"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	# If there is a variable to store the output, use it. Otherwise just
	# return the output through echoing.
	[ -n "${outp}" ] && eval ${outp}="'${dir}'" || echo "${dir}"

	return 0
}

# I2C GPIO: Set the value of I2C GPIO Expander GPIOs.
#
# Parameters:
#  $1: The expander's I2C bus number (dec, hex, octal accepted).
#  $2: The expander's I2C address (dec, hex, octal accepted).
#  $3: The GPIO value.
#  $4: The expander's GPIO base number (starting at zero).
#  $5: [opt] The number of GPIOs. If not specified, 1 is assumed.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
i2c_gpio_val_set() {
	local bus="${1}"
	local addr="${2}"
	local val="${3}"
	local base="${4}"
	local num="${5}"

	local gpio_base gpio_num start

	local ret=0
	local f="${FUNCNAME[0]}"

	# Get the I2C GPIO data.
	i2c_gpio_data_get "${bus}" "${addr}" gpio_base gpio_num || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get I2C GPIO data (${ret})"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	# Validate the GPIOs base/num against valid gpio_base/gpio_num.
	_i2c_gpio_range_check "${base}" "${num}" "${gpio_base}" "${gpio_num}" \
		|| ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" \
			"Invalid GPIOs '${base}..$((${base} + ${num} - 1)) (${num})'"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi
	start=$((${gpio_base} + ${base}))

	gpio_val_set "${val}" "${start}" "${num}" || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Could not set value in all '${num}' GPIOs (${ret})"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	return 0
}

# I2C GPIO: Get the value of I2C GPIO Expander GPIO.
#
# Parameters:
#  $1: The expander's I2C bus number (dec, hex, octal accepted).
#  $2: The expander's I2C address (dec, hex, octal accepted).
#  $3: The expander's GPIO number (starting at zero).
#  $4: [opt] Variable to store the output.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
#  <echo>|<outp>: The GPIO value.
i2c_gpio_val_get() {
	local bus="${1}"
	local addr="${2}"
	local num="${3}"
	local outp="${4}"

	local gpio_base gpio_num start _value

	local ret=0
	local f="${FUNCNAME[0]}"

	# Get the I2C GPIO data.
	i2c_gpio_data_get "${bus}" "${addr}" gpio_base gpio_num || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get I2C GPIO data (${ret})"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	# Validate the GPIOs base/num against valid gpio_base/gpio_num.
	_i2c_gpio_range_check "${num}" "1" "${gpio_base}" "${gpio_num}" \
		|| ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Invalid GPIO '${num}'"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi
	start=$((${gpio_base} + ${num}))

	gpio_val_get "${start}" _value || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get GPIO '${num}' value (${ret})"
		log_err "${f}" "Bus ${bus}, Address ${addr}"
		return ${ret}
	fi

	# If there is a variable to store the output, use it. Otherwise just
	# return the output through echoing.
	[ -n "${outp}" ] && eval ${outp}="'${_value}'" || echo "${_value}"

	return 0
}
