#!/bin/sh
# This script install all configuration defaults if current configurations
# do not exist in the board.
#
# To clean all current configurations and install defaults, pass -f option.
#
# If you want to mantain any current configuration pass them in a list after
# options.

BARIX_CONFIG_PATH="/barix/local/config"
BARIX_DEFAULTS_PATH="/barix/config/defaults"

usage() {
    echo -e "Usage: $0 [−h] [−f] EXC1 EXC2" 1>&2
    echo -e "\t-h: print this help" 1>&2
    echo -e "\t-f: flush all configurations before install defaults" 1>&2
    echo -e "\tEXC: configurations to be mantained when installing defaults" 1>&2
}

set_password_with_path() {
  # note: realm must match the one set in /etc/config.d/httpd
  realm="Barix Login"
  if [ -f /barix/info/DEFAULT_UI_PASSWORD ]; then
    default_ui_password=$(cat /barix/info/DEFAULT_UI_PASSWORD)
  else
    default_ui_password=$(jq -r .applicationPassword /mnt/shadow/config.json)
  fi
  hash_admin=`echo -n "admin:$realm:$default_ui_password" | md5sum | cut -b -32`
  hash_user=`echo -n "user:$realm:user" | md5sum | cut -b -32`

  # get directory of input full file name
  SRC=$1
  BASE=${SRC##*/} 
  DIR=${SRC%$BASE}  

  rm -f $1
  mkdir -p $DIR 
  echo "admin:$realm:${hash_admin}" >> $1
  echo "user:$realm:${hash_user}" >> $1
}

while getopts "hf" opt; do
    echo $opt
    case "$opt" in
        h)
            usage
            exit 0
            ;;
        f)
            CONFIG_FLUSH=1
            ;;
        ?)
            usage
            exit 1
            ;;
    esac
done

shift $((OPTIND-1))
CONFIGS_KEEP="$@"

if [ ! -d "${BARIX_CONFIG_PATH}" ]; then
    echo "Barix local config does not exist..."	
    mkdir -p "${BARIX_CONFIG_PATH}"
    CONFIG_FLUSH=1
fi

if [ -n "${CONFIG_FLUSH}" ]; then
    if [ -d "${BARIX_DEFAULTS_PATH}" ]; then
        # Backup all configurations on the list
        for file in ${CONFIGS_KEEP}; do
            mv -f ${BARIX_CONFIG_PATH}/${file} /tmp
        done
        rm -rf ${BARIX_CONFIG_PATH}/*
        cp -r ${BARIX_DEFAULTS_PATH}/* ${BARIX_CONFIG_PATH}/
        # Restore all backuped configurations
        for file in ${CONFIGS_KEEP}; do
            mv -f /tmp/${file} ${BARIX_CONFIG_PATH}/
        done
        
        cp -r /barix/config/etc/* /barix/local/etc/ 2>/dev/null || :
        cp /etc/qiba-version /barix/local
        if [ ! -f /barix/local/etc/passwd.hash ]; then
            set_password_with_path /barix/local/etc/passwd.hash 2>&1 >/dev/null
        fi
        if [ ! -f /barix/local/etc/.passwd ]; then
            set_password_with_path /barix/local/etc/.passwd 2>&1 >/dev/null
        fi
        echo "Syncing ..."
        sync
    else
        echo "Cannot find default Barix configuration ..."
    fi
else
    echo "Keeping existing local config ..."
fi

