#!/bin/bash

UCI=/sbin/uci
INIT=/sbin/init
UCI_DEFAULTS_DIR=/barix/config/defaults
UCI_TEMPLATES_DIR=/barix/config/templates
UCI_LOCALCFG_DIR=/barix/local/config
UCI_OPTS="-q"

CONFIG_LOCK=/var/run/.service_restart_lock
SERVICES_TO_RESTART=/var/run/.service_restart_list

# prints a value of a parameter without LF
# arguments: 
#	$1 = file.section.parameter
function cfg_print_param() {
	if [ $# -lt 1 ] ; then echo "Config Error: cfg_print_value() called with too few parameters."; return 1 ; fi

	val=`"$UCI" $UCI_OPTS get "$1"`
	echo -n "$val"
}

# prints a value of a parameter without LF escaping "'" javascript delimiters
# arguments:
#	$1 = file.section.parameter
function cfg_print_param_2js() {
  val=$(cfg_print_param "$1")
  val=${val//\'/\\\'}
  val=${val//\"/&quot;}
  echo -n "$val"
}

# prints a value of a parameter or a default value without LF
# arguments: 
#	$1 = file.section.parameter
#	$2 = default value
function cfg_print_param_default() {
	if [ $# -lt 2 ] ; then echo "Config Error: cfg_print_value_default() called with too few parameters."; return 1 ; fi

	val=`"$UCI" $UCI_OPTS get "$1"`
        if [ "$val" == "" ]; then
          val="$2"
        fi
	echo -n "$val"
}

# sets parameter to a value
# arguments: 
#	$1 = file.section.parameter
#	$2 = value
function cfg_set_param() {
	if [ $# -lt 2 ] ; then echo "Config Error: cfg_set_value() called with too few parameters."; return 1 ; fi

	"$UCI" $UCI_OPTS set "$1=$2"
}

# restarts system services of the current runlevel that are listed in $SERVICES_TO_RESTART
# this function is called from /etc/init.d/rcC
# if a service is not in the current runlevel rcX.d folder or vice versa the service is not restarted
# to restart all services pass "*" in $SERVICES_TO_RESTART
function cfg_restart_services_internal() {
	restart_all=0

	# called without a lock file -> avoid mess-up and rather exit
	if [ ! -f "$CONFIG_LOCK" ] ; then 
		echo "Error: cfg_restart_services_internal called without a lock file."
		echo "This is an internal function of the configuration subsystem."
		echo "Call cfg_restart_services_async() to restart services."
		return 1
	fi

	# remove lock if we're killed
	trap 'rm -f "$CONFIG_LOCK" "$SERVICES_TO_RESTART"; return $?' INT TERM EXIT

	# called without a list of services lock file -> avoid mess-up and rather exit
	if [ ! -f "$SERVICES_TO_RESTART" ] ; then
		echo "Error: cfg_restart_services_internal called without a list of services."
		rm -f "$CONFIG_LOCK";
		return 1
	fi

	# parse $SERVICES_TO_RESTART file
	# store services into an associative array to search in the next loop
	declare -A params
	while read service ; do
		# special handling of "*" to restart all
		if [ "$service" = "*" ] ; then restart_all=1 ; break; fi
		params["$service"]="restart"
	done < "$SERVICES_TO_RESTART"

	# go through /barix/config/templates/rc.d scripts in the numbered order and stop
	# services that have been listed as parameters
	for script in ${UCI_TEMPLATES_DIR}/rc.d/S* ; do
		script_name=${script#${UCI_TEMPLATES_DIR}/rc.d/S[0-9][0-9]*}
		if [ $restart_all = 1 -o "${params[$script_name]}"X = "restartX" ] ; then
			echo "Stopping service: [ $script_name ] "
			"$script" stop > /dev/null
		fi
	done

	# go through /barix/config/templates/rc.d scripts in the numbered order and start
	# services that have been listed as parameters
	for script in ${UCI_TEMPLATES_DIR}/rc.d/S* ; do
		script_name=${script#${UCI_TEMPLATES_DIR}/rc.d/S[0-9][0-9]*}
		if [ $restart_all = 1 -o "${params[$script_name]}"X = "restartX" ] ; then
			echo "Starting service: [ $script_name ] "
			"$script" start > /dev/null
		fi
	done

	rm -f "$CONFIG_LOCK" 
#"$SERVICES_TO_RESTART"
	trap - INT TERM EXIT

	return 0
}


# schedules restart of system services that are listed as arguments of the function
# call this function after committing configuration, restoring defaults or uploading config
# uses init "c" and /etc/init.d/rcC to perform the service restart - this is just an entry point
# if an argument is not in the current runlevel rcX.d folder or vice versa the service is not restarted
# $1... $N - system service to be restarted
# to restart all services pass "*" as $1
#
# returns:
#	0 = success, service restart scheduled
#	1 = failed, already restarting
function cfg_restart_services_async() {

	# nothing to restart -> exit
	if [ $# -lt 1 ] ; then return 0 ; fi

	# acquire lock
	if ( set -o noclobber; echo "$$" > "$CONFIG_LOCK" ) 2> /dev/null; then
		# remove lock if we're killed
		trap 'rm -f "$CONFIG_LOCK"; return $?' INT TERM EXIT

		# write all parameters into $SERVICES_TO_RESTART file
		> "$SERVICES_TO_RESTART"
		for service in "$@" ; do 
			echo "$service" >>"$SERVICES_TO_RESTART"
		done

		# trigger init to restart services
		# www-data (lighttpd) cannot run init, so it uses sudo
		if [ "$(id -u)" == "0" ] ; then
			"$INIT" c
		else
			sudo "$INIT" c
		fi

		# remove trap and exit
		# the lock file is removed by cfg_restart_services_internal
		trap - INT TERM EXIT
		return 0
	else
		#echo "Failed to acquire $lockfile. Held by $(cat $lockfile)"
		return 1
	fi

}


# returns 0 (true) if system services are still restarting due to cfg_restart_services_async call
# 1 (false) otherwise, i.e. system is fully operational now
function cfg_restarting_services() {
	if  [ -f "$CONFIG_LOCK" ] ; then
		return 0
	else
		return 1
	fi
}

# commits configuration parameters and restarts the respective services
# 
# 10-mar-2014 - Angelo - introduced dependency system
#
function cfg_save() {
	# first create a list of services that have been changed
	# remove + or - from parameters that have been added or removed
	uci_changes=`"$UCI" $UCI_OPTS changes`
	changes=`echo "$uci_changes" | cut -d "." -f 1 | sed "s/^[-+]//" | tr "\\n" " "`

	# "uci changes" output is a space separated string,
	# as: "network.eth0.ip=192.168.0.104 network.eth0.mask=255.255.255.0"
	params=(`echo $uci_changes | tr " " "\n"`)
	for param in "${params[@]}"
	do
		field_name=`echo "${param%=*}"`
		for file in /barix/config/deps/*
		do
			while read line
			do
				valid_line=`grep -v '^#' <<< "$line"`
				if [ -z "$valid_line" ]; then continue; fi
				grep_output=`grep -F "$valid_line" <<< "$field_name"`
				if [ -n "$grep_output" ]; then
					changes=`echo "$changes ${file##*/}"`
				fi
			done < "$file"
		done	
	done

	# commit changes
	"$UCI" $UCI_OPTS commit

	# restart services that have changed
	cfg_restart_services_async $changes
}


# dump configuration database to stdout
function cfg_dump_database() {
	"$UCI" $UCI_OPTS export
}

# restore database from file $1
# does not restart services
function cfg_restore_database() {
	if [ $# -lt 1 ] ; then echo "Config Error: cfg_restore_database() called with too few parameters."; return 1 ; fi

	"$UCI" $UCI_OPTS import < "$1"
	cfg_restart_services_async "*"
}

# restore factory defaults
# restart all affected services 
function cfg_restore_defaults() {
	declare -A services
	for file in $UCI_DEFAULTS_DIR/* ; do
		"$UCI" $UCI_OPTS import < "$file"
		service=`basename "$file"`
		services[$service]=yes
		# set permisions so the webui can access them
		if [[ $EUID -eq 0 ]]; then
			chown www-data:www-data "$UCI_LOCALCFG_DIR/$service"
			chmod 755 "$UCI_LOCALCFG_DIR/$service"
        	else
			sudo /bin/chmod 755 "$UCI_LOCALCFG_DIR/$service"
		fi
	done
# delete also the stored copy of the player.cfg from the flash
# for now do it always but in the future we might need a hook
# detecting the "active" application name, and do it only for ferdinand
    if [[ $EUID -ne 0 ]]; then
        sudo /bin/rm -f /barix/app-data/com.barix.ferdinand/player.cfg
    else
        rm -f /barix/app-data/com.barix.ferdinand/player.cfg
    fi
	cfg_restart_services_async "${!services[@]}"
}

# restore soft factory defaults (all defaults except network and fallback server)                                                                                           
# restart all affected services                                                                                                                                             
function cfg_restore_soft_defaults() {                                                                                                                                      
        declare -A services                                                                                                                                                 
        for file in $UCI_DEFAULTS_DIR/*  
        do
#  skip the network settings                                                                                                                                                 
                if basename "$file" | grep -Fxq network                                                                                                                     
                then 
                        continue
                fi
                if basename "$file" | grep -Fxq application
                then
# for the application settings skip the fallback server settings 
                        cat "$file" | grep -iv "option config_" | grep . > /tmp/.tmpdef
# and store the existing fallback server settings
                        TMP_OPT=$($UCI show application.ferdinand.config_server | awk -F= '{print $2}')
                        echo -e "\toption config_server\t'$TMP_OPT'" >> /tmp/.tmpdef 
                        TMP_OPT=$($UCI show application.ferdinand.config_dir | awk -F= '{print $2}')
                        echo -e "\toption config_dir\t'$TMP_OPT'" >> /tmp/.tmpdef 
                        TMP_OPT=$($UCI show application.ferdinand.config_port | awk -F= '{print $2}')
                        echo -e "\toption config_port\t'$TMP_OPT'" >> /tmp/.tmpdef 
                        TMP_OPT=$($UCI show application.ferdinand.config_protocol | awk -F= '{print $2}')
                        echo -e "\toption config_protocol\t'$TMP_OPT'" >> /tmp/.tmpdef 
                        TMP_OPT=$($UCI show application.ferdinand.config_username | awk -F= '{print $2}')
                        echo -e "\toption config_username\t'$TMP_OPT'" >> /tmp/.tmpdef 
                        TMP_OPT=$($UCI show application.ferdinand.config_password | awk -F= '{print $2}')
                        echo -e "\toption config_password\t'$TMP_OPT'" >> /tmp/.tmpdef 
                        "$UCI" $UCI_OPTS import < /tmp/.tmpdef                                           
                else                                                                                                                                                        
                        "$UCI" $UCI_OPTS import < "$file"                                                                                                                   
                fi                                                                                                                                                          
                service=`basename "$file"`                                                                                                                                  
                services[$service]=yes
        done
# delete also the stored copy of the player.cfg from the flash
# for now do it always but in the future we might need a hook
# detecting the "active" application name, and do it only for ferdinand
        if [[ $EUID -ne 0 ]]; then
            sudo /bin/rm -f /barix/app-data/com.barix.ferdinand/player.cfg
        else
            rm -f /barix/app-data/com.barix.ferdinand/player.cfg
        fi
        
        /usr/bin/barix-restore-defaults.sh soft
                       
        cfg_restart_services_async "${!services[@]}"                                                                                                                        
}

#---------------------------
# assumes a bool parameter, returns bash true or false according to the bool parameter value
# can be used in expressions
# arguments: 
#	$1 = file.section.parameter
cfg_bool_is_true() {
	if [ $# -lt 1 ] ; then echo "Config Error: cfg_bool_is_true() called with too few parameters."; return 1 ; fi

	val=`"$UCI" $UCI_OPTS get "$1"`
	if [ "$val" = "yes" -o "$val" = "true" -o "$val" = "on" ] 
		then return 0 
		else return 1
	fi
}


# compares a string parameter with a value passed as argument $2
# returns bash true if the string equal or false if they differ
# can be used in expressions
# arguments: 
#	$1 = file.section.parameter
#	$2 = value to compare
cfg_string_compare() {
	if [ $# -lt 2 ] ; then echo "Config Error: cfg_string_compare() called with too few parameters."; return 1 ; fi

	val=`"$UCI" $UCI_OPTS get "$1"`
	if [ "$val" = "$2" ] 
		then return 0 
		else return 1
	fi
}

# returns true if the configuration parameter $1 has been changed, false otherwise
function cfg_has_changed() {
	if "$UCI" $UCI_OPTS changes | grep "$1" > /dev/null ; then 
		return 0
	else
		return 1
	fi
}

