import json
import logging
import os
from typing import Tuple, List, Optional, Callable, Dict, Any
from uci import Uci, UciExceptionNotFound, UciException
from .UciExceptions import InvalidUCIException

logger = logging.getLogger('flask-backend')


class UciCli:

    @staticmethod
    def get_value(config: str, section: str or None=None, option: str or None=None) -> str:
        """
        Get the value from a provided uci
        If only config is provided, it will return a dictionary of dictionaries where:
            - first level keys are section keys and the values are dictionaries with section content.
            - second level keys are options and lists from that section. If it is an option, the value is either a string and or a tuple containing strings.
        If only config and section is provided, it will return a dictionary with all options and lists in that section.
        If config, section and option are provided, then it will return either a string or a tuple of strings if it consists on a list.
        If the requested config or section are not found, the returned value is None.
        If the requested option doesn't exist, an empty value will be returned.
        If any other exception is thrown, the returned value is None.
        :param config
        :param section: optional
        :param option: optional
        """

        value = None

        uci_cli = Uci()

        # get only config
        if config is not None and section is None and option is None:
            try:
                value = uci_cli.get(config)
            except UciExceptionNotFound:
                logger.debug(f"Config \"{config}\" NOT FOUND")
            except Exception as e:
                logger.error(f"Error occurred while getting config \"{config}\" value: {e}")
                raise e
        # get config and section
        elif config is not None and section is not None and option is None:
            try:
                value = uci_cli.get_all(config, section)
            except UciExceptionNotFound:
                logger.debug(f"section \"{config}.section{section}\" NOT FOUND")
            except Exception as e:
                logger.error(f"Error occurred while getting section \"{config}.{section}\" value: {e}")
                raise e
        # get config, section and option
        elif config is not None and section is not None and option is not None:
            try:
                value = uci_cli.get(config, section, option)
            except UciExceptionNotFound:
                # logger.debug(f"UCI \"{config}.{section}.{option}\" NOT FOUND")
                value = ''
            except Exception as e:
                logger.error(f"Error occurred while getting uci \"{config}.{section}.{option}\" value: {e}")
                raise e
        else:
            raise InvalidUCIException

        return value

    @staticmethod
    def filter_unchanged_ucis(list_of_ucis: Dict[str, Any]) -> Dict[str, Any]:
        """
        Filters a dictionary of proposed UCI settings, returning only those
        that are different from the currently saved value.

        This method uses a single UCI instance for all read operations
        to avoid repeatedly calling Uci().
        """

        changed_ucis = {}
        uci_cli = None
        try:
            uci_cli = Uci()

            for uci_path, new_value in list_of_ucis.items():
                params = uci_path.split('.')
                if len(params) != 3:
                    logger.warning(f"Skipping malformed UCI path: {uci_path}")
                    continue

                config, section, option = params
                current_value_str: str | None = None

                try:
                    current_value = uci_cli.get(config, section, option)

                    if isinstance(current_value, (list, tuple)):
                        # Always update lists/tuples
                        changed_ucis[uci_path] = new_value
                        continue

                    current_value_str = str(current_value)

                except UciExceptionNotFound:
                    # Distinguish non-existent option from empty string
                    current_value_str = None
                except Exception as e:
                    logger.error(f"Error getting UCI value {uci_path}: {e}")
                    raise e

                # Normalize new value for comparison
                if isinstance(new_value, bool):
                    str_new_value = 'true' if new_value else 'false'
                else:
                    str_new_value = str(new_value)

                # Include in changed_ucis if value differs or option is missing
                if current_value_str != str_new_value:
                    changed_ucis[uci_path] = new_value

        except Exception as e:
            logger.error(f"Error during UCI value filtering: {e}")
            raise e

        return changed_ucis



    @staticmethod
    def get_value_as_boolean(config: str, section: str or None = None, option: str or None = None) -> bool:
        value = UciCli.get_value(config, section, option)
        return UciCli.to_boolean(value)


    @staticmethod
    def get_value_as_integer(config: str, section: str or None = None, option: str or None = None) -> int:
        value = UciCli.get_value(config, section, option)
        return UciCli.to_integer(value)


    @staticmethod
    def get_value_as_json(config: str, section: str or None = None, option: str or None = None) -> Any:
        value = UciCli.get_value(config, section, option)
        return UciCli.to_json(value)


    @staticmethod
    def to_boolean(value: Any) -> bool:
        if isinstance(value, bool):
            return value
        if isinstance(value, str):
            return value.lower() in ('true', '1', 'yes', 'on', 't', 'y')
        if isinstance(value, (int, float)):
            return bool(value)
        return False


    @staticmethod
    def boolean_to_string(value: Any) -> str:
        """
        Convert a boolean-like value to its string representation.
        """
        bool_value = UciCli.to_boolean(value)
        return 'true' if bool_value else 'false'


    @staticmethod
    def to_integer(value: Any) -> int:
        if isinstance(value, int):
            return value
        if isinstance(value, float):
            return int(value)
        if isinstance(value, str):
            try:
                return int(value)
            except ValueError:
                try:
                    return int(float(value))
                except ValueError:
                    return 0
        return 0


    @staticmethod
    def integer_to_string(value: Any) -> str:
        if isinstance(value, str):
            try:
                int(value)  # Check if it's a valid integer string
                return value  # Return the original string if it is
            except ValueError:
                pass
        int_value = UciCli.to_integer(value)
        return str(int_value)


    @staticmethod
    def to_json(value: Any) -> Any:
        if isinstance(value, (dict, list)):
            return value
        if isinstance(value, str):
            try:
                return json.loads(value)
            except json.JSONDecodeError:
                return None
        return None


    @staticmethod
    def json_to_string(value: Any) -> str:
        """
        Convert a JSON-like value to its string representation.
        """
        json_value = UciCli.to_json(value)
        if json_value is None:
            return ''  # or 'null', depending on your preference
        try:
            return json.dumps(json_value, ensure_ascii=False)
        except TypeError:
            return ''  # or 'null', for non-JSON-serializable objects


    @staticmethod
    def set_value(config, section=None, option=None, value=None, verify=False, uci_cli=None, log_changes_callback=None):
        """
        Set uci value
        If option is not passed, defaults to None and the entire section value will be set. Value must be a dictionary.
        If option is passed, value must be a string or a list, and option value will be set, instead of section.
        If verify is True, it will check if value changed before setting. If it didn't change, it won't set the new value.
        If verify is not passed, it will default to False, and the value will be set, even if it didn't changed.
        uciCli is also an argument because if one uses a new uciCli for each value set before commiting, previous changes will be lost. If uciCli is none, a new one will be used.
        :param config:
        :param section:
        :param option:
        :param value:
        :param verify:
        :param uci_cli:
        :param log_changes_callback:
        """

        can_set = True
        uci_set = False
        curr_value = None

        if uci_cli is None:
            uci_cli = Uci()

        value = UciCli.convert_bool_value_to_string(value)
        # logger.debug(f"Setting {config}.{section}.{option}={value}")

        if verify:
            try:
                curr_value = UciCli.get_value(config, section, option)
                if isinstance(value, dict):
                    if curr_value != '' and curr_value is not None:
                        can_set = False
                        curr_value_json = json.loads(curr_value)
                        for newKey in value.keys():
                            if newKey not in curr_value_json.keys():
                                can_set = True
                                break
                            else:
                                if curr_value_json[newKey] != value[newKey]:
                                    can_set = True
                                    break
                        for oldKey in curr_value_json.keys():
                            if oldKey not in value.keys():
                                can_set = True
                                break

                else:
                    if curr_value == str(value):
                        # logger.debug(f"Value from uci '{config}.{section}.{option}' hasn't changed, no need to set")
                        can_set = False
            except Exception as e:
                raise e

        if can_set:
            try:
                if isinstance(value, dict):
                    value = json.dumps(value)
                logger.debug(f"Setting UCI '{config}.{section}.{option}', value '{value}'")
                uci_cli.set(config, section, option, str(value))
                try:
                    if log_changes_callback is not None:
                        log_changes_callback(config, section, option, curr_value, str(value))
                except Exception as e:
                    logger.error(e)
                uci_set = True
            except UciException: # invalid uci
                logger.error(f"Invalid UCI '{config}.{section}.{option}'")
                raise UciException(f"'{config}.{section}.{option}'")
            except Exception as e:
                logger.error(f"Error while setting uci '{config}.{section}.{option}' with value '{value}' : {str(e)}")
                raise e

        return uci_cli, uci_set


    @staticmethod
    def get_value_as_boolean(config: str, section: str or None = None, option: str or None = None) -> bool:
        value = UciCli.get_value(config, section, option)
        return UciCli.to_boolean(value)


    @staticmethod
    def get_value_as_integer(config: str, section: str or None = None, option: str or None = None) -> int:
        value = UciCli.get_value(config, section, option)
        return UciCli.to_integer(value)


    @staticmethod
    def get_value_as_json(config: str, section: str or None = None, option: str or None = None) -> Any:
        value = UciCli.get_value(config, section, option)
        return UciCli.to_json(value)


    @staticmethod
    def to_boolean(value: Any) -> bool:
        if isinstance(value, bool):
            return value
        if isinstance(value, str):
            return value.lower() in ('true', '1', 'yes', 'on', 't', 'y')
        if isinstance(value, (int, float)):
            return bool(value)
        return False


    @staticmethod
    def boolean_to_string(value: Any) -> str:
        """
        Convert a boolean-like value to its string representation.
        """
        bool_value = UciCli.to_boolean(value)
        return 'true' if bool_value else 'false'


    @staticmethod
    def to_integer(value: Any) -> int:
        if isinstance(value, int):
            return value
        if isinstance(value, float):
            return int(value)
        if isinstance(value, str):
            try:
                return int(value)
            except ValueError:
                try:
                    return int(float(value))
                except ValueError:
                    return 0
        return 0


    @staticmethod
    def integer_to_string(value: Any) -> str:
        if isinstance(value, str):
            try:
                int(value)  # Check if it's a valid integer string
                return value  # Return the original string if it is
            except ValueError:
                pass
        int_value = UciCli.to_integer(value)
        return str(int_value)


    @staticmethod
    def to_json(value: Any) -> Any:
        if isinstance(value, (dict, list)):
            return value
        if isinstance(value, str):
            try:
                return json.loads(value)
            except json.JSONDecodeError:
                return None
        return None


    @staticmethod
    def json_to_string(value: Any) -> str:
        """
        Convert a JSON-like value to its string representation.
        """
        json_value = UciCli.to_json(value)
        if json_value is None:
            return ''  # or 'null', depending on your preference
        try:
            return json.dumps(json_value, ensure_ascii=False)
        except TypeError:
            return ''  # or 'null', for non-JSON-serializable objects


    @staticmethod
    def convert_bool_value_to_string(value):
        """
        Converts a python boolean value to a string.
        For example, True -> "true"; False -> "false".
        If value is a dict, it will iterate over it is content and convert the values of the keys if applies.
        @param value: value to convert
        """
        if isinstance(value, bool):
            if value:
                value = "true"
            else:
                value = "false"
        return value


    @staticmethod
    def convert_value_from_string_to_type_value(value):
        """
        Converts a string to the respective python value.
        It can return a boolean, an integer, a string or a dictionary
        @param value: string value to convert
        """
        if isinstance(value, str):
            #check if is bool
            if value == "true":
                value = True
            elif value == "false":
                value = False
            #check if is a number
            elif value.isdigit():
                value = int(value)
            else:
                #check if is dict
                try:
                    value_json = json.loads(value)
                    for elem in value_json:
                        value_json[elem] = UciCli.convert_value_from_string_to_type_value(value_json[elem])
                    value = value_json
                except Exception:
                    # value is string
                    pass
        elif isinstance(value, dict):
            for elem in value:
                value[elem] = UciCli.convert_value_from_string_to_type_value(value[elem])
        # bool, int and list pass right through
        return value


    @staticmethod
    def commit_ucis(configs_list, uci_cli):
        """
        Commit a set of uci's configs to turn permanent their value changes
        @param configs_list: set of configs to commit
        @param uci_cli: uciCli to use to commit the list of configs provided.
        """
        for config in configs_list:
            try:
                # logger.debug(f"Commiting config {config}")
                uci_cli.commit(config)
            except Exception as e:
                # should never happen!!
                # return_msg = {"error_code": COMMIT_UCI_ERROR_CODE, "msg": f"It was not possible to commit config {config}"}
                logger.error(f"Committing config {config} failed: {e}")
                # error code: Internal Server Error
                raise e


    @staticmethod
    def format_uci_tuple_into_string(uci_tuple):
        """
        Format a tuple containing the uci's config, section and option into a string with those parameters contatenated with '.'
        For example, ('application', 'audio', 'volume') -> 'application.audio.volume'
        @param uci_tuple: tuple containing the uci's config, section and option
        """
        uci_string = uci_tuple[0] + "." + uci_tuple[1] + "." + uci_tuple[2]
        return uci_string


    @staticmethod
    def get_all_configs():
        """
        Get all uci's configs configured in the device.
        It will get all the names of the files that exist in the directory /barix/config/current (which correspond to the configs being used in the device)
        """
        try:
            mypath = "/barix/config/current"
            config_names = [f for f in os.listdir(mypath) if os.path.isfile(os.path.join(mypath, f))]
        except Exception as e:
            logger.error(e)
            raise e
        else:
            return config_names


    @staticmethod
    def uci_config_exists(config: str) -> bool:
        """
        Check if a config exists
        :param config:
        :return:
        """
        uci_cli = Uci()
        try:
            return uci_cli.get(config) is not None
        except UciExceptionNotFound as uci_ex:
            return False


    @staticmethod
    def set_uci_configs(list_of_ucis: Dict[str, str], commit=True, log_changes_callback: Optional[Callable[[str, str, str], None]] = None) -> List[
        Tuple[str, str, str]]:
        """
        @param list_of_ucis: dictionary with the list of ucis to set, where the keys are the uci's name and the value is the value to store on the uci. At the
        moment, only uci's names with the following syntax are allowed: 'config.section.option'.
        If restart services required, need to call the `Utils.restart_related_services(ucis_configured)` after.
        @param commit: (optional) flag that indicates if the changes should be commited or not. If True or not provided, changes will be commited
        :param log_changes_callback:
        :param log_changes_callback: Optional callback
        """
        ucis_configured = []  # in case of failure, to revert
        configs_to_commit = set()  # to commit ucis
        uci_cli = None
        try:
            for uci in list_of_ucis:
                params = uci.split('.')
                if len(params) == 3:
                    # config = params[0], section = params[1], option = params[2], value = jsonUcis[uci]
                    uci_cli, uci_set = UciCli.set_value(params[0], params[1], params[2], list_of_ucis[uci], True, uci_cli, log_changes_callback)
                    if uci_set:
                        ucis_configured.append((params[0], params[1], params[2]))
                        configs_to_commit.add(params[0])

            # commit uci changes?
            if commit:
                # uci changes must be committed
                UciCli.commit_ucis(configs_to_commit, uci_cli)

        except UciException as e:
            raise e
        except Exception as e:
            raise e
        else:
            return ucis_configured

    @staticmethod
    def reset_section(config:str, section:str, value="section") -> None:
        """
        Function to delete section and setting with new value
        @param config string Specify config to be selected
        @param section string Specify section to be reseted
        @param value: (optional) "section" Specify final value for selected config.section
        """
        _uci = Uci()
        _uci.delete(config, section)
        _uci.commit(f"{config}.{section}")

        _uci.set(config, section, value)
        _uci.commit(f"{config}.{section}")
