import os
import json
from typing import List, Dict, Any, Tuple
import logging

logger = logging.getLogger('flask-backend')


RESTART_CONFIG_PATH = "config/restart-config.json"

class Uci2ServiceMap:
    _instance = None
 
    _initial_configs = {
        "time_settings.timezone":               ["S20timezone-handler"],
        "time_settings.ntp_server1":            ["S20ntpd"],
        "time_settings.ntp_server2":            ["S20ntpd"],
        "time_settings.ntp_server3":            ["S20ntpd"],

        "network":                              ["reboot"],  # TODO check if reboot is required when changing network
        "dropbear":                             ["dropbear"],

        "avahi":                                ["webui"],
        "httpd":                                ["webui"],

        "network.eth0":                         ["sonic_ip"],
    }

    _services = {
        "S21avahi-daemon":        {"priority": 21,   "command": ["/etc/rc5.d/S21avahi-daemon restart"]},
        "dropbear":               {"priority": 99,   "command": ["/barix/hooks/application/dropbear.hook stop", "/barix/hooks/application/dropbear.hook start"]},
        "S20ntpd":                {"priority": 20,   "command": ["/etc/rc5.d/S20ntpd restart"]},
        "ussi-controller":        {"priority": 99,   "command": ["/barix/hooks/application/ussi-controller.hook stop", "/barix/hooks/application/ussi-controller.hook start"]},
        "S20timezone-handler":    {"priority": 20,   "command": ["/etc/rc5.d/S20timezone-handler restart"]},
        "sonic_ip":               {"priority": 99,   "command": [""]},
        "webui":                  {"priority": 95,   "command": ["/usr/bin/barix-wd --pid-file=/var/run/webui.pid --restart"]},

        "backend_flask":          {"priority": 99,   "command": ["barix-wd --pid-file=/var/run/webui.pid --restart"]},
    }


    def __new__(cls):
        if cls._instance is None:
            cls._instance = super(Uci2ServiceMap, cls).__new__(cls)
            cls._instance._initialized = False
        return cls._instance


    def __init__(self):
        if self._initialized:
            return
        self._initialized = True
        self._configurations = self._initial_configs.copy()

        # Read the specific services and configs to reload....
        if os.path.exists(RESTART_CONFIG_PATH):
            try:
                with open(RESTART_CONFIG_PATH, 'r') as f:
                    data = json.load(f)        
                    merged_configs = self._initial_configs

                    # Merge new config JSON
                    for config in data["configs"]:
                        for key, services in config.items():
                            if key not in merged_configs:
                                merged_configs[key] = services
                            else:
                                for service in services:
                                    if service not in merged_configs[key]:
                                        merged_configs[key].append(service)
                    
                    self._configurations = dict(merged_configs)
                    self._services.update(data['services'])

                logger.info(f"Loaded custom restart config from {RESTART_CONFIG_PATH}")
            except json.JSONDecodeError:
                logger.error(f"Error decoding JSON from {RESTART_CONFIG_PATH}. Using only base restart config.", exc_info=True)
            except IOError as e:
                logger.error(f"Error reading {RESTART_CONFIG_PATH}: {e}. Using only base restart config.", exc_info=True)
        else:
            logger.info(f"{RESTART_CONFIG_PATH} not found. Using only base restart config.")


    def get_affected_services_for_ucis(self, uci_list) -> List[str]:
        """
        Process a list of config strings and return corresponding service names.
        :param uci_list: List of strings in the format "config.section.option"
        :return: List of service names corresponding to matching configurations
        """

        services_to_restart = []
        for uci in uci_list:
            # CASE UCIS ARE A ENTIRE STRING
            uci_parts = uci.split('.')

            uci_1 = ""
            uci_2 = ""
            uci_3 = ""

            # TODO - improve, this should be done recursively until the len of the uci!

            if len(uci_parts) > 0:
                uci_1 = uci_parts[0]

            if len(uci_parts) > 1:
                uci_2 = uci_parts[0] + '.' + uci_parts[1]

            if len(uci_parts) > 2:
                uci_2 = uci_parts[0] + '.' + uci_parts[1] + '.' + uci_parts[2]
            
            # "config"
            if uci_1 in self._configurations:
                # services_to_restart.append(self._configurations[uci_1])
                services_to_restart += self._configurations[uci_1]

            # "config.section"
            if uci_2 in self._configurations:
                # services_to_restart.append(self._configurations[uci_2])
                services_to_restart += self._configurations[uci_2]

            # "config.section.option"
            if uci_3 in self._configurations:
                # services_to_restart.append(self._configurations[uci_3])
                services_to_restart += self._configurations[uci_3]

        return services_to_restart


    def sort_services_by_priority(self, services_list):
        try:
            return sorted(services_list, key=lambda service: self._services[service]["priority"])
        except Exception as e:
            logger.error(e)
            return services_list


    def get_service_command(self, service):
        return self._services[service]["command"]
