import { PostLogoutResponseBody } from "@custom-types/base/authentication.types";
import { GetAllLogsResponseBody, GetLogsResponseBody } from "@custom-types/base/logs.types";
import { GetMqttSettingsResponseBody } from "@custom-types/base/mqtt.types";
import { HttpMethod, HttpResponse } from "@custom-types/base/request-response.types";
import { request } from "@utils/base/generic-request";

/**
 * API endpoints for the Base API.
 */
enum BaseApiEndpoint {
    Login = "/login",
    Logout = "/logout",
    CaCerts = "/cacerts",
    SystemLogs = "/logs/system",
    Logs = "/log-list",
    MqttSettings = "/mqtt/settings",
    Sdf = "/sdf/<instance_name>",
    Settings = "/settings",
    FirmwareUpdate = "/settings/fw-update",
    ChangePassword = "/settings/change-password",
    SystemStatus = "/system/status",
    SystemAbout = "/system/about",
    SystemInfo = "/system/update/info",
    SystemDefaults = "/system/defaults",
    SystemUpdate = "/system/update",
    SystemReboot = "/system/reboot",
    Ucis = "/ucis"
}

/**
 * Utility class for interacting with the device Base API.
 */
export class BaseApi {
    // --- Authentication ---
    /** Log in to the portal. */
    static login(body: Cypress.RequestBody): Cypress.Chainable<Cypress.Response<undefined>> {
        return request(HttpMethod.POST, BaseApiEndpoint.Login, body);
    }
    /** Log out of the portal. */
    static logout(): Cypress.Chainable<Cypress.Response<PostLogoutResponseBody>> {
        return request(HttpMethod.POST, BaseApiEndpoint.Logout);
    }

    // --- Certificates ---
    /**
     * Get Certificate Authority certificates.
     * @deprecated Backend returns 404; this endpoint may be deprecated.
     */
    static getCaCertificates(): Cypress.Chainable<Cypress.Response<any>> {
        return request(HttpMethod.GET, BaseApiEndpoint.CaCerts);
    }

    // --- Logs ---
    /** Get all logs. */
    static getAllLogs(): Cypress.Chainable<Cypress.Response<GetAllLogsResponseBody>> {
        return request(HttpMethod.GET, BaseApiEndpoint.Logs);
    }

    /** Get system logs. */
    static getSystemLogs(): Cypress.Chainable<Cypress.Response<GetLogsResponseBody>> {
        return request(HttpMethod.GET, BaseApiEndpoint.SystemLogs);
    }

    // --- MQTT ---
    /** Get MQTT configuration settings. */
    static getMqttSettings(): Cypress.Chainable<Cypress.Response<GetMqttSettingsResponseBody>> {
        return request(HttpMethod.GET, BaseApiEndpoint.MqttSettings);
    }

    // --- UCIS ---
    /**
     * Get UCIS.
     * @deprecated Backend returns 404; this endpoint may be deprecated.
     */
    static getUcis(): Cypress.Chainable<Cypress.Response<HttpResponse>> {
        return request(HttpMethod.GET, BaseApiEndpoint.Ucis);
    }

    // --- System Information ---
    /**
     * Get system information.
     * @param query Optional query parameters.
     */
    static getSystemStatus(): Cypress.Chainable<Cypress.Response<HttpResponse>> {
        return request(HttpMethod.GET, BaseApiEndpoint.SystemStatus);
    }

    static getSystemAbout(): Cypress.Chainable<Cypress.Response<HttpResponse>> {
        return request(HttpMethod.GET, BaseApiEndpoint.SystemAbout);
    }

    static getSystemInfo(): Cypress.Chainable<Cypress.Response<HttpResponse>> {
        return request(HttpMethod.GET, BaseApiEndpoint.SystemInfo);
    }

    /** Get firmware update settings. */
    static getFirmwareUpdateSettings(): Cypress.Chainable<Cypress.Response<HttpResponse>> {
        return request(HttpMethod.GET, BaseApiEndpoint.FirmwareUpdate);
    }

    /** Update system settings. */
    static updateSettings(body: Cypress.RequestBody): Cypress.Chainable<Cypress.Response<HttpResponse>> {
        return request(HttpMethod.PUT, BaseApiEndpoint.Settings, body);
    }

    /** Reset system to default settings. */
    static resetSystemDefaults(): Cypress.Chainable<Cypress.Response<HttpResponse>> {
        return request(HttpMethod.POST, BaseApiEndpoint.SystemDefaults);
    }

    /**
     * Get the current state of the system update process.
     * Possible states: "clear", "uploading", "installing", "rebooting", "error".
     */
    static getSystemUpdate(): Cypress.Chainable<Cypress.Response<HttpResponse>> {
        return request(HttpMethod.GET, BaseApiEndpoint.SystemUpdate);
    }

    /** Reboot the device. */
    static reboot(body: Cypress.RequestBody): Cypress.Chainable<Cypress.Response<HttpResponse>> {
        return request(HttpMethod.POST, BaseApiEndpoint.SystemReboot, body);
    }

    /** Change user password. */
    static changePassword(body: Cypress.RequestBody): Cypress.Chainable<Cypress.Response<HttpResponse>> {
        return request(HttpMethod.POST, BaseApiEndpoint.ChangePassword, body);
    }
}
