import { PostLoginRequestBody } from "@custom-types/base/authentication.types";
import { HttpResponse } from "@custom-types/base/request-response.types";
import { webUiPassword, webUiUsername } from "@environment";
import { BaseApi } from "@models/base/base.api";
import { clearCache } from "@utils/base/reset-device";

// This might be deprecated/not in use
describe.skip('Update settings', () => {
    beforeEach(() => {
        clearCache();
        console.log(`%c${Cypress.currentTest.title}:`, 'font-weight: bold');
    });

    it('Update settings on a logged in account without providing a body', () => {
        const requestBody: PostLoginRequestBody = {
            username: webUiUsername,
            password: webUiPassword
        }

        BaseApi.login(requestBody).then((response) => {
            expect(response.status).to.be.equal(200);
            BaseApi.updateSettings({}).then((response: Cypress.Response<HttpResponse>) => {
                expect(response.status).to.be.equal(400);
            });
        });
    });

    it('Update settings on a non logged in account', () => {
        BaseApi.updateSettings({}).then((response: Cypress.Response<HttpResponse>) => {
            expect(response.status).to.be.equal(401);
        });
    });

    it('Set UCI configurations', () => {
        const requestBody: PostLoginRequestBody = {
            username: webUiUsername,
            password: webUiPassword
        }

        BaseApi.login(requestBody).then((response) => {
            expect(response.status).to.be.equal(200);

            const requestBody = { ucis: { 'application.fw_update.auto': false }, commit: true };
            BaseApi.updateSettings(requestBody).then((response: Cypress.Response<HttpResponse>) => {
                expect(response.status).to.be.equal(200);
            });
        });
    })


    it('Update Date and Time using the right format', () => {
        const requestBody: PostLoginRequestBody = {
            username: webUiUsername,
            password: webUiPassword
        }

        BaseApi.login(requestBody).then((response) => {
            expect(response.status).to.be.equal(200);

            const requestBody = { timeSettings: '2025-05-05 15:00:00' };

            BaseApi.updateSettings(requestBody).then((response: Cypress.Response<HttpResponse>) => {
                expect(response.status).to.be.equal(200);
            });
        });
    })

    it('Update Date and Time using the wrong format (string)', () => {
        const requestBody: PostLoginRequestBody = {
            username: webUiUsername,
            password: webUiPassword
        }

        BaseApi.login(requestBody).then((response) => {
            expect(response.status).to.be.equal(200);

            const requestBody = { timeSettings: '15:00:00 2025-05-05' };

            BaseApi.updateSettings(requestBody).then((response: Cypress.Response<HttpResponse>) => {
                expect(response.status).to.be.equal(400);
            });
        });
    })

    it('Update Date and Time using the wrong format (object)', () => {
        const requestBody: PostLoginRequestBody = {
            username: webUiUsername,
            password: webUiPassword
        }

        BaseApi.login(requestBody).then((response) => {
            expect(response.status).to.be.equal(200);

            const requestBody = { timeSettings: { date: '05/05/2025', time: '12:00:00 PM' } };

            BaseApi.updateSettings(requestBody).then((response: Cypress.Response<HttpResponse>) => {
                expect(response.status).to.be.equal(400);
            });
        });
    })


    it('Update invalid settings', () => {
        const requestBody: PostLoginRequestBody = {
            username: webUiUsername,
            password: webUiPassword
        }

        BaseApi.login(requestBody).then((response) => {
            expect(response.status).to.be.equal(200);

            const requestBody = { notAValidSetting: true, notAValidSettingEither: "true" };

            BaseApi.updateSettings(requestBody).then((response: Cypress.Response<HttpResponse>) => {
                expect(response.status).to.be.equal(400);
            });
        });
    })


    it('Set UCI and Date and Time with valid configurations', () => {
        const requestBody: PostLoginRequestBody = {
            username: webUiUsername,
            password: webUiPassword
        }

        BaseApi.login(requestBody).then((response) => {
            expect(response.status).to.be.equal(200);

            const requestBody = {
                timeSettings: '2025-05-05 15:00:00',
                ucis: { 'application.fw_update.auto': false }, commit: true
            };

            BaseApi.updateSettings(requestBody).then((response: Cypress.Response<HttpResponse>) => {
                expect(response.status).to.be.equal(200);
            });
        });
    });


    it('Set Web UI password by providing a wrong old password', () => {
        const requestBody: PostLoginRequestBody = {
            username: webUiUsername,
            password: webUiPassword
        }

        BaseApi.login(requestBody).then((response) => {
            expect(response.status).to.be.equal(200);

            const requestBody = {
                pwdSettings: { pwd_old: 'Wrong_Password', pwd_set: 'New_Password' }
            };

            BaseApi.updateSettings(requestBody).then((response: Cypress.Response<HttpResponse>) => {
                expect(response.status).to.be.equal(400);
            });
        });
    });


    it('Set Web UI password by without providing the passwords', () => {
        const requestBody: PostLoginRequestBody = {
            username: webUiUsername,
            password: webUiPassword
        }

        BaseApi.login(requestBody).then((response) => {
            expect(response.status).to.be.equal(200);

            const requestBody = {
                pwdSettings: {}
            };

            BaseApi.updateSettings(requestBody).then((response: Cypress.Response<HttpResponse>) => {
                expect(response.status).to.be.equal(400);
            });
        });
    });

    it('Set Web UI password providing the correct passwords', () => {
        const requestBody: PostLoginRequestBody = {
            username: webUiUsername,
            password: webUiPassword
        }

        BaseApi.login(requestBody).then((response) => {
            expect(response.status).to.be.equal(200);

            const newPassword = 'New_Password';
            const requestBody = {
                pwdSettings: { pwd_old: webUiPassword, pwd_set: newPassword }
            };

            BaseApi.updateSettings(requestBody).then((response: Cypress.Response<HttpResponse>) => {
                expect(response.status).to.be.equal(200);

                // After changing the password the device should allow to login with the new password
                BaseApi.login({ ...requestBody, password: newPassword }).then((response) => {
                    expect(response.status).to.be.equal(200);
                });
            });
        });
    });
});
