import { PostLoginRequestBody } from "@custom-types/base/authentication.types";
import { HttpResponse } from "@custom-types/base/request-response.types";
import { webUiPassword, webUiUsername } from "@environment";
import { BaseApi } from "@models/base/base.api";
import { decodeJwtPayload } from "@utils/base/decode-jwt";
import { clearCache } from "@utils/base/reset-device";

describe('Login', () => {
    beforeEach(() => {
        clearCache();
        console.log(`%c${Cypress.currentTest.title}:`, 'font-weight: bold');
    });

    it('Login with valid credentials', () => {
        const requestBody: PostLoginRequestBody = {
            username: webUiUsername,
            password: webUiPassword
        }

        BaseApi.login(requestBody).then((response) =>{
            expect(response.status).to.be.equal(200);

            const cookies = response.headers["set-cookie"];
            expect(cookies, 'Set-Cookie header should exist').to.be.an('array').and.have.length.greaterThan(0);

            // Extract session JWT from the first cookie
            const firstCookie = cookies[0];
            const match = RegExp(/session=([^;]+)/).exec(firstCookie);
            expect(match, 'Session JWT should be present in the cookie').to.not.be.null;

            const sessionJwt = match![1];

            // Decode and assert JWT payload
            const payload = decodeJwtPayload(sessionJwt);
            expect(payload, 'Decoded JWT payload should have required keys').to.include.all.keys('username', 'sessionID');
        });
    });

    it('Login with invalid credentials', () => {
        const requestBody: PostLoginRequestBody = {
            username: "invalid_username",
            password: "invalid_password"
        }

        BaseApi.login(requestBody).then((response) =>{
            expect(response.status).to.be.equal(401);
        });
    });

    it('Login without providing password', () => {
        const requestBody = {
            username: webUiUsername
        }

        BaseApi.login(requestBody).then((response) =>{
            expect(response.status).to.be.equal(400);
        });
    });

    it('Login without providing username', () => {
        const requestBody = {
            password: webUiPassword
        }

        BaseApi.login(requestBody).then((response) =>{
            expect(response.status).to.be.equal(400);
        });
    });
});
