import logging
import json
import os
from typing import List, Dict, Any

from common_lib.lib.configuration import config

logger = logging.getLogger('flask-backend')


def get_all_configs () -> List[Dict[Any, Any]]:
    """
    Retrieves all log configurations by combining static and dynamic configuration sources.

    This function performs the following:
    1. Loads static log configurations from the primary configuration path
    2. Optionally loads dynamic configurations from additional JSON files in a dynamic directory
    3. Combines all configurations into a single list

    Returns:
        list: A combined list of all log configuration objects from both static and dynamic sources

    Note:
        - Dynamic configurations are loaded only if 'log_dynamic_conf_path' exists in config
        - Invalid JSON files in dynamic directory are skipped, with logging
        - The function ensures all configurations are merged into a unified list
    """
    all_logs = []
    logs_conf_path = config["log_conf_path"]
    all_logs.extend(get_configs(logs_conf_path))

    #  Load dynamic configuration:
    if "log_dynamic_conf_path" in config:
        dynamic_logs_conf_path = config["log_dynamic_conf_path"]
        if os.path.isdir(dynamic_logs_conf_path):
            for filename in os.listdir(dynamic_logs_conf_path):
                if filename.endswith('.json'):
                    file_path = os.path.join(dynamic_logs_conf_path, filename)
                    try:
                        configs = get_configs(file_path)
                        all_logs.extend(configs)
                    except Exception as e:
                        logger.warning(f"Failed to load dynamic config {filename}: {str(e)}")
    return all_logs


def get_configs (config_path) -> List[Dict[Any, Any]]:
    """
    Loads and extracts log configurations from a specified JSON configuration file.

    This function reads a JSON configuration file and extracts log configuration objects
    from the "config" key. It handles multiple error scenarios.

    Args:
        config_path (str): Path to the JSON configuration file to be loaded

    Returns:
        list: A list of log configuration objects extracted from the file

    Raises:
        Exception: Always raises an exception with message "Error getting configurations!"
                  if the file doesn't exist or cannot be processed successfully

    Note:
        - The function expects the JSON file to have a "config" key containing an array
        - Detailed error logging is provided for troubleshooting
    """
    logs_info = []
    if os.path.exists(config_path):
        try:
            with open(config_path, 'r') as f:
                logs_file = json.load(f)
            if "config" in logs_file:
                logs_info.extend(logs_file["config"])
            logger.info(f"Loaded custom log list from {config_path}")
            return logs_info
        except json.JSONDecodeError:
            logger.error(f"Error decoding JSON from {config_path}. Using only base logs.", exc_info=True)
        except IOError as e:
            logger.error(f"Error reading {config_path}: {e}. Using only base logs.", exc_info=True)
    else:
        logger.error(f"File path doesn't exists. Path: {config_path}")
    raise "Error getting configurations!"
