from flask import request, Response, jsonify
import logging
import zoneinfo
import sys
import json
import subprocess
from common_lib.lib.fw_update import update_firmware, get_fw_update_status, stop_fw_image_upload
from common_lib.lib.exceptions import InvalidUploadedFileError, FileTooLargeError
# from common_lib.lib.system_info import get_system_info
from common_lib.lib.FileUtils import FileUtils
from common_lib.rest.web.web_base_routes import web_base_bp
from common_lib.rest.web.midleware.login_required import login_required
from common_lib.rest.web.midleware.update_state_validator import update_state_validator
from common_lib.lib.system_info import \
    get_mac_addr, \
    get_hw_type_id, \
    get_device_type, \
    get_device_type_id, \
    get_device_alias, \
    get_ipam_type, \
    get_ipam_type_id, \
    get_kernel_ver, \
    get_local_time, \
    get_capabilities, \
    get_network_settings, \
    get_up_time, \
    get_licenses, \
    get_mass_storage_devices, \
    get_log_lines, \
    get_fw_ver, \
    get_root_fs_compilation_date, \
    _find_and_update_param

from common_lib.uci.UciCli import UciCli



logger = logging.getLogger('flask-backend')

try:
    product_info = FileUtils.get_json('config/product-info.json', logger)
    DEVICE_NAME = product_info.get("device_name", "Barix Device")
    logger.info(f"Successfully loaded device name: {DEVICE_NAME}")
except Exception as e:
    logger.error(f"Failed to load product info from 'product-info.json': {e}")
    DEVICE_NAME = "Barix Device"


@web_base_bp.route('/system/factory-defaults', methods=['POST'])
@login_required
@update_state_validator
def web_handle_reset_to_defaults():
    """
    Reset the device's configurations to the default ones.
    """
    try:
        cmd = "/usr/bin/barix-restore-defaults.sh soft > /dev/null 2>&1"
        subprocess.run(cmd, shell=True)
        return '', 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500


@web_base_bp.route('/system/update', methods=['POST'])
@login_required
@update_state_validator
def web_handle_fw_image_update():
    try:
        update_firmware(request.files)
        return '', 200
    except InvalidUploadedFileError as e:
        logger.error(e, exc_info=True)
        return jsonify({"message": str(e)}), 400
    except FileTooLargeError as e:
        logger.error(e, exc_info=True)
        return jsonify({"message": str(e)}), 413
    except Exception as e:
        logger.error(e, exc_info=True)
        stop_fw_image_upload()
        return '', 500


@web_base_bp.route('/system/update', methods=['GET'])
@login_required
def web_handle_get_fw_update_status():
    """
    This retrieve all the steps status of firmware update.
    Note: Doesn't need the @update_state_validator
    """
    update_status = get_fw_update_status()
    return jsonify({"message": update_status}), 200


@web_base_bp.route('/system/update/info', methods=['GET'])
@login_required
@update_state_validator
def application_get_system_update_info():
    try:
        data = {
            "fw_version":       get_fw_ver(),
            "kernelVer":       get_kernel_ver(),
            "rootFSDate":       get_root_fs_compilation_date(),
        }

        return jsonify(data), 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500



@web_base_bp.route('/system/timezones', methods=['GET'])
@login_required
def get_available_timezones():
    try:
        tz_list = sorted(list(zoneinfo.available_timezones()))

        return jsonify(tz_list), 200

    except Exception as e:
        logger.error(f'Error obtaining timezones list: {e}', exc_info=True)
        return jsonify({"error": "It wasn't possible to obtain the timezones list"}), 500




@web_base_bp.route('/system/update-timezone-options', methods=['POST'])
@login_required
def update_timezone_options():
    """
    Updates the timezone options in the SDF config file
    with the list from zoneinfo.
    """
    SDF_CONFIG_FILE_PATH = '/barix/apps/webui/common_lib/sdf/basic-settings-sdf-config.json'
    try:
        # 1. Get the list of timezones
        tz_list = sorted(list(zoneinfo.available_timezones()))

        # 2. Format them for the JSON structure
        new_options = [{"value": tz, "label": tz} for tz in tz_list]

        # 3. Read the existing JSON file
        try:
            with open(SDF_CONFIG_FILE_PATH, 'r', encoding='utf-8') as f:
                sdf_config = json.load(f)
        except FileNotFoundError:
            logger.error(f'SDF config file not found at: {SDF_CONFIG_FILE_PATH}')
            return jsonify({"error": "Configuration file not found."}), 500
        except json.JSONDecodeError:
            logger.error(f'Failed to decode JSON from: {SDF_CONFIG_FILE_PATH}')
            return jsonify({"error": "Configuration file is corrupted or not valid JSON."}), 500


        # 4. Find and update the timezone parameter
        config_data_list = sdf_config.get("config", [])
        was_updated = _find_and_update_param(config_data_list, "timezone", new_options)

        if not was_updated:
            logger.warning("Could not find 'timezone' parameter in SDF config.")
            return jsonify({"error": "Timezone parameter not found in config file."}), 404

        # 5. Write the updated data back to the file
        with open(SDF_CONFIG_FILE_PATH, 'w', encoding='utf-8') as f:
            # Use indent=2 to keep the file human-readable
            json.dump(sdf_config, f, indent=2)

        logger.info(f"Successfully updated timezone options. Added {len(new_options)} timezones.")
        return jsonify({
            "message": "Timezone options updated successfully.",
            "timezones_added": len(new_options)
        }), 200

    except Exception as e:
        logger.error(f'Error updating timezone options: {e}', exc_info=True)
        return jsonify({"error": "An internal server error occurred."}), 500



@web_base_bp.route('/system/reboot', methods=['POST'])
@login_required
@update_state_validator
def web_reboot_request():
    try:
        subprocess.run('reboot')
        return '', 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500


@web_base_bp.route('/system/status', methods=['GET'])
@login_required
@update_state_validator
def application_get_system_status():
    try:
        data = {
            "device": {
                "Mac_addr":         get_mac_addr(),
                # "HW_type":          "M400",
                "HW_type_ID":       get_hw_type_id(),
                "type":             get_device_type(),
                "type_ID":          get_device_type_id(),
                "IPAM_type":        get_ipam_type(),
                "IPAM_type_ID":     get_ipam_type_id(),
                "kernelVer":        get_kernel_ver(),
                "local_time":       get_local_time(),
                "uptime":           get_up_time(),
                "fw_version":       get_fw_ver(),
                "alias":            get_device_alias(),
                "device_name":      DEVICE_NAME
            },
            "capabilities": get_capabilities(),
            "network":      get_network_settings(),
            "licenses":     get_licenses(),
            "massStorageDev": get_mass_storage_devices()
        }

        return jsonify(data), 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500

@web_base_bp.route('/system/status/config', methods=['GET'])
@login_required
@update_state_validator
def get_system_status_config():
    filepath = '/barix/apps/webui/common_lib/adf/system-status-config.json'
    try:
        with open(filepath, "r") as f:
            config = json.load(f)
    except Exception as e:
        logger.error(f'Error reading SDF config file "{filepath}": {e}', exc_info=True)

    if not config:
        return '', 500

    return jsonify(config), 200


# NOTE: Retrieve device system information that are available with a non login state
@web_base_bp.route('/system/about', methods=['GET'])
@update_state_validator
def application_get_system_about():
    try:
        data = {
            "fw_version":       get_fw_ver(),
            "alias":            get_device_alias(),
            "device_name":      DEVICE_NAME,
        }

        return jsonify(data), 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500


@web_base_bp.route('/system/open-licenses', methods=['GET'])
@login_required
def application_get_licenses():
    try:
        system_licenses = []
        file_system_licenses = "/barix/info/licenses-report.csv"
        try:
            with open(file_system_licenses) as file:
                file_content = file.read()
                file_content = file_content.replace('"', '')
                file_lines = file_content.splitlines()
                for line in file_lines[1:]:
                    license_parts = line.split(',')
                    # open_source_licenses.append(license_parts)
                    if license_parts[2].lower() == "closed":
                        continue
                    system_licenses.append(license_parts)

        except Exception as e:
            logger.error(f"Error trying to get system licenses: {e}", exc_info=True)


        web_licenses = []
        file_web_licenses = "/barix/info/angular_licenses.json"
        try:
            with open(file_web_licenses) as file:
                json_data = json.load(file)

                for key in json_data:
                    license = json_data[key]['licenses']

                    if license.lower() == 'unknown':
                        continue

                    if key[0] == '@':
                        key = key[1:]

                    package, version = key.split('@')
                    web_licenses.append([package, version, license])

        except Exception as e:
            logger.error(f"Error trying to get web licenses: {e}", exc_info=True)

        licenses = {
            "System" : system_licenses,
            "Web" : web_licenses,
        }

        return jsonify(licenses), 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500
