import json
import logging
from flask import Response, request, jsonify, session
from typing import Optional

from common_lib.sdf_validators.SdfValueValidator import SdfValueValidator
from common_lib.lib.SdfConfig import SdfConfig
from common_lib.rest.web.controllers.webui_password import check_username_password, store_webui_password, set_web_ui_password
from common_lib.lib.system_info import set_date_and_time, get_device_alias
from common_lib.rest.web.web_base_routes import web_base_bp
from common_lib.rest.web.midleware.login_required import login_required
from common_lib.lib.exceptions import InvalidRequestBodyContentError, InvalidOldPasswordError, SettingDateAndTimeError
from common_lib.rest.web.midleware.update_state_validator import update_state_validator
from common_lib.lib.JsonRules import JsonRules
from common_lib.lib.system_info import set_is_first_login_false
from common_lib.lib.environment_checker import EnvironmentChecker

from common_lib.uci.Utils import Utils
from common_lib.uci.UciCli import UciCli

env_checker = EnvironmentChecker()
logger = logging.getLogger('flask-backend')


@web_base_bp.route('/settings', methods=['PUT'])
@login_required
@update_state_validator
def web_set_device_settings():
    return_msg = {}
    try:
        data = json.loads(request.data)
        logger.debug(f"Device settings to set: {data}")
    except Exception as e:
        logger.error(e, exc_info=True)
        return jsonify({"message": "Invalid request body"}), 400

    if 'pwdSettings' in data.keys():
        try:
            set_web_ui_password(data['pwdSettings'])
        except InvalidRequestBodyContentError:
            return "Invalid request body", 400
        except InvalidOldPasswordError:
            return "Old password is invalid", 400
        except Exception as e:
            logger.error(e, exc_info=True)
            return '', 500


    # TODO - check if this is still used
    """
    if 'ucis' in data.keys():
        if not 'commit' in data.keys():
            logger.error("commit not in uciSettings", exc_info=True)
            return Response("Invalid request body", status=400)
        json_ucis = data['ucis']
        try:
            ucis_configured = UciCli.set_uci_configs(json_ucis, commit=data['commit'])
            # WebUtils.check_if_web_protocol_changed(ucis_configured)
            Utils.restart_related_services(ucis_configured)
            uci_list_set = []
            for uciTuple in ucis_configured:
                uci_list_set.append(UciCli.format_uci_tuple_into_string(uciTuple))

            return_msg = {"ucisSet": uci_list_set}
        # except UciException as e:
        #     return Response("Invalid "+e, status=400)
        except Exception as e:
            logger.error(e, exc_info=True)
            return '', 500
    """

    if 'timeSettings' in data.keys():
        try:
            set_date_and_time(data['timeSettings'])
        except SettingDateAndTimeError:
            return jsonify({"message": "Invalid date and time to set"}), 400
        except Exception:
            return '', 500

    return jsonify(return_msg), 200


@web_base_bp.route('/settings/fw-update', methods=['GET'])
@login_required
@update_state_validator
def web_set_device_fw_update_settings():
    """
    Retrieve necessary info regarding firmware update
    """
    try:
        # TODO - check
        payload = {
            "security.update.enabled": UciCli.get_value('security', 'update',' enabled') == "true",
            "application.fw_update.auto": UciCli.get_value('application',' fw_update',' auto') == "true",
            "application.fw_update.url": UciCli.get_value('application',' fw_update',' url'),
            "application.fw_update.start_time": UciCli.get_value('application',' fw_update',' start_time'),
            "application.fw_update.stop_time": UciCli.get_value('application',' fw_update',' stop_time'),
            "application.fw_update.week_day": UciCli.get_value('application',' fw_update',' week_day'),
        }
        return jsonify(payload), 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500


@web_base_bp.route('/settings/change-password', methods=['POST'])
@login_required
@update_state_validator
def web_change_password():
    data = request.json
    old_password = data.get('oldPassword')
    new_password = data.get('newPassword')
    confirm_password = data.get('confirmPassword')

    # Check for missing fields
    if not all([old_password, new_password, confirm_password]):
        return jsonify({"message": "All fields are required"}), 400

    # Check for password mismatch
    if new_password != confirm_password:
        return jsonify({"message": "New password and confirmation do not match"}), 400

    if not env_checker.is_device_running_develop_mode():
        # Check for strength validation
        validation_error = validate_password(new_password)
        if validation_error:
            return jsonify({"message": validation_error}), 400

    # Authentication and Storage
    # Authenticate old password
    username = session['username']
    if not check_username_password(username, old_password):
        # SECURITY NOTE: Use a generic error message for incorrect password attempts
        # to avoid giving away information about valid usernames.
        return jsonify({"message": "Invalid username or password"}), 401

    # Store new password
    store_webui_password(new_password)

    # Update first login flag
    set_is_first_login_false()

    logger.info(f"Password changed successfully for user: {username}")
    return jsonify({"message": "Password changed successfully"}), 200


def validate_password(password: str) -> Optional[str]:
    """
    Validates the password against strict security requirements.

    Requirements:
    - Minimum 12 characters total.
    - At least one uppercase letter (A-Z).cd ..common_lib
    - At least one lowercase letter (a-z).
    - At least one digit (0-9).
    - At least one special character (e.g., !@#$%^&*()_+).

    Returns:
        A string describing the failure reason, or None if validation passes.
    """
    if len(password) < 12:
        return "Password must be 12 or more characters long."

    # Regular expression lookaheads for strict checking
    # (?=.*[a-z]) : Must contain at least one lowercase letter
    # (?=.*[A-Z]) : Must contain at least one uppercase letter
    # (?=.*\d)    : Must contain at least one digit
    # (?=.*[\W_]) : Must contain at least one special character (non-word character or underscore)
    # [A-Za-z\d\W_]{12,} : Matches the entire string (12+ chars, included for completeness, but length check handles this)

    # We only use the lookaheads combined with the length check for clarity.

    pattern = re.compile(
        r"^(?=.*[a-z])"       # Require one lowercase
        r"(?=.*[A-Z])"       # Require one uppercase
        r"(?=.*\d)"          # Require one digit
        r"(?=.*[\W_])"       # Require one special character (or underscore)
        r".{12,}$"           # Must be 12 or more total characters (handled by length check, but good for regex validation)
    )

    if not pattern.match(password):
        return (
            "Password must contain at least 12 characters, including "
            "one uppercase letter, one lowercase letter, one number, and one special character."
        )

    return None # Validation passed


@web_base_bp.route('/settings/basic', methods=['GET'])
@login_required
@update_state_validator
def application_get_basic_settings():
    try:
        data_device = {}
        data_device['alias'] = get_device_alias()

        # with open("/barix/config/current/network", 'r') as f:
        #     uci_data = f.readlines()

        uci_network = { "network": UciCli.get_value("network")}
        sonic_ip = UciCli.get_value("sonic_ip","general")

        data_ntp = UciCli.get_value("ntp", "source")
        timezone = UciCli.get_value("timezone", "config", "value")
        ssh_enabled = UciCli.get_value("dropbear", "RunCtl", "enable")

        web_protocol = UciCli.get_value("httpd", "webserver", "protocol")

        data_network = {}
        for key in uci_network["network"]:
            interface={
                "protocol":     JsonRules.get_value("network", key, "proto",     uci_network, "dhcp"),
                "ip_address":    JsonRules.get_value("network", key, "ipaddr",    uci_network, ''),
                "netmask":   JsonRules.get_value("network", key, "netmask",   uci_network, ''),
                "gateway":   JsonRules.get_value("network", key, "gateway",   uci_network, ''),
                "dhcpname":  JsonRules.get_value("network", key, "dhcpname",  uci_network, ''),
                "dns_type":  JsonRules.get_value("network", key, "dns_type",  uci_network, "AUTO"),
                "primary_dns":      JsonRules.get_value("network", key, "dns1",      uci_network, ''),
                "alt_dns":      JsonRules.get_value("network", key, "dns2",      uci_network, '')
            }
            interface["protocol"]    = interface["protocol"].lower()
            interface["dns_type"] = interface["dns_type"].lower()

            data_network[key]=interface

        data_device["sonic_ip"] = sonic_ip

        data_network["protocol"] = web_protocol

        json_data = {
            "device" :  data_device,
            # "sonic_ip" : sonic_ip,
            "network" : data_network,
            "time_settings" : {
                "ntp":      data_ntp,
                "timezone": timezone,
            },
            "security": {
                "ssh" : ssh_enabled,
            }
        }

        json_converted = JsonRules.json_to_types(json_data, "./common_lib/rules/basic-settings.rules.json")

        return jsonify(json_converted), 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500


@web_base_bp.route('/settings/basic', methods=['POST'])
@login_required
@update_state_validator
def application_set_basic_settings():
    try:
        list_ucis = {}
        data = json.loads(request.data)

        #
        # SDF values validation
        #
        sdf_config = SdfConfig("basic-settings")
        sdf_value_validator = SdfValueValidator(sdf_config.get(), logger)
        sdf_errors = sdf_value_validator.validate(data)
        if sdf_errors:
            logger.info(f"Sdf validation errors: {sdf_errors}")
            retval = {
                "message": "Validation sdf errors found",
                "errors": sdf_errors
            }
            return jsonify(retval), 400

        #
        # validate_rules
        #
        with open("./common_lib/rules/basic-settings.rules.json", 'r') as f:
            data_rules = json.load(f)

        # logger.debug(json.dumps(data_rules, indent=4))

        # Run validation
        validation_errors = JsonRules.validate_rules(data, data_rules)
        if validation_errors:
            retval = {
                "message": "Validation errors found",
                "errors": validation_errors
            }
            return jsonify(retval), 400

        if 'device' in data and 'alias' in data["device"]:
            # _, ucis_configured = jdb_utils.set("device.alias", data["device"]["alias"], True)
            # logger.debug(ucis_configured)
            logger.debug(f"Alias: {data['device']['alias']}")

            # UciCli.set_uci_configs('httpd.webserver.alias', data["device"]["alias"])
            list_ucis["httpd.webserver.alias"] = data["device"]["alias"]

        if 'time_settings' in data:
            if 'timezone' in data['time_settings']:
                list_ucis["timezone.config.value"] = data['time_settings']['timezone']
            if 'ntp' in data['time_settings']:
                list_ucis["ntp.source.server1"] = data['time_settings']['ntp']['server1']
                list_ucis["ntp.source.server2"] = data['time_settings']['ntp']['server2']
                list_ucis["ntp.source.server3"] = data['time_settings']['ntp']['server3']

        if 'security' in data and 'ssh' in data['security']:
            list_ucis["dropbear.RunCtl.enable"] = data['security']['ssh']

        if 'device' in data and 'sonic_ip' in data['device']:
            list_ucis["sonic_ip.general.enabled"] = data['device']['sonic_ip']['enabled']

            if 'volume' in data['device']['sonic_ip']:
                list_ucis["sonic_ip.general.volume"] = data['device']['sonic_ip']['volume']

        # network!!!
        if 'network' in data:
            data_network = data["network"]

            for interface, interface_data in data_network.items():
                if not isinstance(interface_data, dict):
                    continue


                if 'web_protocol' not in data_network:
                    return 'Missing web protocol', 400
                list_ucis[f"httpd.webserver.protocol"] = data_network['web_protocol']

                if 'protocol' not in data_network[interface]:
                    return 'Missing protocol', 400
                list_ucis[f"network.{interface}.proto"] = data_network[interface]['protocol']

                if data_network[interface]['protocol'] == 'static':
                    try:
                        list_ucis[f"network.{interface}.ipaddr"]   = data_network[interface]['ip_address']
                        list_ucis[f"network.{interface}.gateway"]  = data_network[interface]['gateway']
                        list_ucis[f"network.{interface}.netmask"]  = data_network[interface]['netmask']
                        list_ucis[f"network.{interface}.dns_type"] = 'manual'
                        list_ucis[f"network.{interface}.dns1"]     = data_network[interface]['primary_dns']
                        list_ucis[f"network.{interface}.dns2"]     = data_network[interface]['alt_dns']
                    except:
                        return f"Missing {interface} parameter", 400

        logger.debug("Set ucis:")
        for uci in list_ucis:
            logger.debug(uci)

        try:
            logger.debug(f"Total potential UCI changes: {len(list_ucis)}")
            changed_ucis = UciCli.filter_unchanged_ucis(list_ucis)
            logger.debug(f"Actual UCI changes to be applied: {len(changed_ucis)}")

        except UciException as e:
            logger.error(f"UCI error during value filtering: {e}", exc_info=True)
            return "Internal server error (UCI read failed)", 500

        services_to_restart = []

        if not changed_ucis:
            logger.debug("No UCI changes detected. Returning success.")
            return jsonify({'services_restarting': services_to_restart}), 200

        changed_uci_tuples = UciCli.set_uci_configs(changed_ucis)

        logged_ucis = [".".join(t) for t in changed_uci_tuples]

        logger.debug("ucis changed:")
        logger.debug(logged_ucis)

        services_to_restart = Utils.restart_related_services(logged_ucis)

        logger.info("Saved settings values")

        return jsonify({'services_restarting': services_to_restart}), 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500






# import os
# import time
# from flask import secrets
#
# TODO
# API_KEY_FILE_PATH
# readFromJSONFile(path)
#
# @web_base_bp.route('/settings/api', methods=['GET'])
# @login_required
# @update_state_validator
# def web_get_api_key():
#     api_key_dict = {}
#     try:
#         if os.path.isfile(API_KEY_FILE_PATH):
#             api_key_dict = readFromJSONFile(API_KEY_FILE_PATH)
#     except Exception as e:
#         logger.error("An error occurred while reading from file {}:{}".format(API_KEY_FILE_PATH, e))
#         raise e
#     else:
#         return api_key_dict
# def web_generate_api_key():
#     try:
#         api_key = secrets.token_urlsafe(32)
#         timestamp = time.time()
#         return {"api-key":api_key, "timestamp": timestamp}
#     except Exception as e:
#         logger.error(e)
#         raise e
# def web_ipac_store_api_key(api_key_dict):
#     try:
#         with open(API_KEY_FILE_PATH, 'w') as outfile:
#             json.dump(api_key_dict, outfile)
#     except Exception as e:
#         logger.error("An error occurred while writing into file {}:{}".format(API_KEY_FILE_PATH, e))
#         raise e
#
#
# @web_base_bp.route('/api', methods=['POST'])
# @login_required
# @update_state_validator
# def web_ipac_update_api_key():
#     try:
#         api_key_dict = web_generate_api_key()
#         web_ipac_store_api_key(api_key_dict)
#         return Response(json.dumps(api_key_dict), status=200)
#     except Exception as e:
#         logger.error(e)
#         return Response(status=500)
