import logging
import os
from flask import send_file, make_response, jsonify
from common_lib.lib.system_info import download_all_system_logs, get_log_lines
from common_lib.rest.web.midleware.login_required import login_required
from common_lib.rest.web.midleware.update_state_validator import update_state_validator
from common_lib.rest.web.utils.logs import get_all_configs
from common_lib.rest.web.web_base_routes import web_base_bp

logger = logging.getLogger('flask-backend')


@web_base_bp.route('/logs/system', methods=['GET'])
@login_required
@update_state_validator
def web_get_system_log():
    try:
        return jsonify({"message": get_log_lines("/var/log/messages", 60)}), 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500


@web_base_bp.route('/logs/download', methods=['GET'])
@login_required
@update_state_validator
def web_download_all_logs():
    try:
        filename = download_all_system_logs()
        file_path = os.path.join('/mnt/data', filename)

        # TODO - verify if this can be removed (A. Maia)
        response = make_response(send_file(file_path, as_attachment=True, download_name=filename))
        response.headers['Cache-Control'] = 'no-store'
        return response

        # return send_file(os.path.join('/mnt/data', filename), as_attachment=True, mimetype='application/gzip')

    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500


@web_base_bp.route('/log-list', methods=['GET'])
@login_required
@update_state_validator
def web_get_available_logs():
    try:
        logs_info = get_all_configs()
        keys = [x["label"] for x in logs_info]
        return jsonify(keys), 200

    except Exception as e:
        logger.error(f"Error retrieving available log names: {e}", exc_info=True)
        return '', 500


@web_base_bp.route('/logs/<string:log_name>', methods=['GET'])
@login_required
@update_state_validator
def web_get_specific_log(log_name):
    """
    Fetches and returns the content of a specific log file identified by its 'name' (label).
    The log name must be present in the combined list of base and custom logs.
    Optional query parameter 'lines' to limit the number of lines returned.
    """
    try:
        logs_info = get_all_configs()
        log_path = None
        for x in logs_info:
            if "label" in x and "path" in x and x["label"] == log_name:
                log_path = x["path"]

        if not log_path:
            logger.warning(f"Requested log name '{log_name}' not found in available log list.")
            return jsonify({"error": f"Log '{log_name}' not found or configured."}), 404

        try:
            if not os.path.isfile(log_path):
                logger.error(f"Configured log path '{log_path}' for '{log_name}' is not a file or does not exist.")
                # NOTE: At device initial state, some logs are not created yet. For this
                return jsonify({"message": ""}), 200
                # return jsonify({"error": f"Configured path for log '{log_name}' is not a valid log file."}), 404

            if not os.access(log_path, os.R_OK):
                logger.error(f"Permission denied to read log file: {log_path}")
                return jsonify({"error": f"Permission denied to access log '{log_name}'."}), 403

            logger.info(f"Served log '{log_name}' from '{log_path}'")
            return jsonify({"message": get_log_lines(log_path, 60)}), 200

        except FileNotFoundError:
            logger.error(f"Log file not found at path: {log_path}", exc_info=True)
            return jsonify({"error": f"Log file for '{log_name}' not found on the server."}), 404
        except Exception as e:
            logger.error(f"Error fetching log file '{log_path}' for '{log_name}': {e}", exc_info=True)
            return '', 500

    except Exception as e:
        logger.error(f"Unexpected error in web_get_specific_log for '{log_name}': {e}", exc_info=True)
        return '', 500
