import json
import logging
from flask import session, request, make_response, jsonify
from common_lib.rest.web.controllers.webui_password import check_username_password
from common_lib.rest.web.midleware.login_required import get_session_key, login_required
from common_lib.rest.web.web_base_routes import web_base_bp
from common_lib.lib.system_info import get_is_first_login

logger = logging.getLogger('flask-backend')


@web_base_bp.route('/login', methods=['POST'])
def login():
    try:
        data = json.loads(request.data)
        if 'username' not in data or 'password' not in data:
            return jsonify({"message": "Invalid login form!"}), 400

        username = data['username']
        password = data['password']
        if check_username_password(username, password):
            session_id = get_session_key()
            session['sessionID'] = session_id
            session['username'] = username
            session.modified = True

            logger.info(f"Logged in: '{username}' (Session ID: {session_id})")
            return jsonify({"message": "Success", "firstLogin": get_is_first_login()}), 200
        else:
            logger.error("Can't login. Invalid username or password!", exc_info=True)
            return jsonify({"message": "Invalid username or password"}), 401
    except Exception as e:
        logger.error(f"Error while checking login: {e}", exc_info=True)
        return '', 500


@web_base_bp.route('/logout', methods=['POST'])
@login_required
def logout():
    try:
        username = session.get('username', 'Unknown User')
        session_id = session.get('sessionID', 'Unknown Session')

        session.pop('username', None)
        session.pop('sessionID', None)

        # Clear the entire session
        session.clear()

        response = make_response({"message": "Logged out successfully"}, 200)
        response.set_cookie('session', '', expires=0)

        logger.info(f"Logged out: '{username}' (Session ID: {session_id})")
        return response

    except Exception as e:
        logger.error(f"Exception during logout for user '{session.get('username', 'Unknown User')}': {str(e)}")
        return jsonify({"message": "An error occurred during logout"}), 500
