from functools import wraps
from flask import session, jsonify
import logging
import os
import uuid

from common_lib.lib.configuration import config

logger = logging.getLogger('flask-backend')


def login_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        try:
            if 'sessionID' not in session or session['sessionID'] != get_session_key():
                return jsonify({"message":"Login required"}), 401
            else:
                session.modified = True
                return f(*args, **kwargs)
        except Exception as e:
            logger.error(f"Error while checking login: {e}", exc_info=True)
            return '', 500

    return decorated_function


def get_session_key():
    """
    Get the session key.
    Session Key is used to validate logins.
    Reads the file SESSION_KEY_FILE_PATH, which is where the session key is stored.
    If the file doesn't exist or if it is empty, generates a new session key.
    Returns the session key (string).
    """
    try:
        # if file does not exist, create it
        if not os.path.isfile(config['session_key_file_path']):
            session_key = generate_new_session_key()
        else: # file exists
            f = open(config['session_key_file_path'], 'r')
            session_key = f.read()
            if session_key == '': # file is empty
                session_key = generate_new_session_key()
        return session_key
    except Exception as e:
        raise e


def generate_new_session_key():
    """
    Generate a new session key.
    Session Key is used to validate logins.
    A unique ID is generated and stored in the file SESSION_KEY_FILE_PATH.
    """
    session_key = uuid.uuid4().hex
    try:
        f = open(config['session_key_file_path'], 'w')
        f.write(session_key)
        f.close()
        return session_key
    except Exception as e:
        logger.error(f"Error while writing in file {config['session_key_file_path']}: {e}", exc_info=True)
        raise e
