import logging
import hashlib
from typing import Dict


from common_lib.lib.exceptions import InvalidRequestBodyContentError, InvalidOldPasswordError
from common_lib.rest.web.midleware.login_required import generate_new_session_key
from common_lib.lib.configuration import config

logger = logging.getLogger('flask-backend')


def set_web_ui_password(pwd_settings: Dict[str, str]) -> None :
    """
    Sets a new password for the webUI.
    @param pwd_settings: dictionary containing the old password (pwd_old) and the new one (pwd_set). The password values are in plain text(!).
    The old password may not be provided if the login with password is disabled for the webUI.
    If required but not provided, or provided but not required, it will raise an InvalidRequestBodyContentError exception.
    If required and provided but invalid, it will raise an InvalidOldPasswordError exception.
    For the new password, if not provided or empty, the webUI's login with password will be disabled. Otherwise, the new password will be set.
    """

    if 'pwd_old' in pwd_settings.keys():
            try:
                is_valid = curr_password_is_valid(pwd_settings['pwd_old'])
            except Exception as e:
                raise e

            if not is_valid:
                logger.error("Old password is wrong", exc_info=True)
                raise InvalidOldPasswordError()
    else:
        logger.error("pwd_old not in pwdSettings", exc_info=True)
        raise InvalidRequestBodyContentError()

    # change or enable webui password
    try:
        store_webui_password(pwd_settings['pwd_set'])
    except Exception as e:
        raise e


def check_username_password(username: str, password: str) -> bool:
    """
    Validate pair username and password.
    :param username:
    :param password:
    :return:
    """
    try:
        return curr_password_is_valid(password, username=username)
    except Exception as e:
        logger.error(e, exc_info=True)
        return False

def curr_password_is_valid(
    passwd: str,
    passwd_file: str = config["webui_passwd_file_path"],
    username: str = config["default_webui_login_username"],
    realm: str = config["default_webui_login_realm"]
) -> bool:
    """
    Checks if the password provided matches the current one.
    @param passwd: password to test
    @param passwd_file: filepath of the file where current password is stored. If not provided, default path is WEBUI_PASSWD_FILE_PATH
    @param username: string containing the username of the corresponding password. If not provided, default path is DEFAULT_WEBUI_LOGIN_USERNAME
    @param realm: string containing the realm used to calculate the digest. If not provided, default path is DEFAULT_WEBUI_LOGIN_REALM
    Calculates the md5 digest of the given username, realm and password. If it matched the one stored in the file, then the password provided is correct:
    returns True. Otherwise, returns False.
    """
    str_to_digest = username + ":" + realm + ":" + passwd
    try:
        digest = hashlib.md5(str_to_digest.encode('utf-8')).hexdigest()
    except Exception as e:
        logger.error(f"Can not calculate digest for the provided password: {e}", exc_info=True)
        raise
    try:
        f = open(passwd_file, 'r')
        content = f.read()
        f.close()
    except Exception as e:
        logger.error(f"Error while reading passwd file contents: {e}", exc_info=True)
        raise e
    try:
        md5sum = ''
        file_lines = content.split('\n')
        for line in file_lines:
            tmp_str = username + ':' + realm + ':'
            if line.startswith(tmp_str):
                md5sum = line.split(':')[2]
                break
        if md5sum != '':
            return digest == md5sum
        else:
            return False
    except Exception as e:
        raise e


def store_webui_password(
    passwd: str,
    passwd_file: str = config["webui_passwd_file_path"],
    username: str = config["default_webui_login_username"],
    realm: str = config["default_webui_login_realm"]
) -> None:
    """
    Store a webUI Password.
    @param passwd: password to store
    @param passwd_file: filepath of the file where to store the password. If not provided, default path is WEBUI_PASSWD_FILE_PATH
    @param username: string containing the username of the corresponding password. If not provided, default path is DEFAULT_WEBUI_LOGIN_USERNAME
    @param realm: string containing the realm used to calculate the digest. If not provided, default path is DEFAULT_WEBUI_LOGIN_REALM
    Calculates the md5 digest of the given username, realm and password: md5digest(username:realm:passwd)
    Then, writes it in filepath with the structure username:realm:md5digest.
    Finally, since the password was changed, current logins must expire and users should login again with the new password. So, generate a new session key.
    """
    str_to_digest = username + ":" + realm + ":" + passwd
    try:
        digest = hashlib.md5(str_to_digest.encode('utf-8')).hexdigest()
    except Exception as e:
        logger.error(f"Can not calculate digest for the provided password : {e}", exc_info=True)
        raise e
    try:
        f = open(passwd_file, 'w')
        str_to_write = username + ":" + realm + ":" + digest
        f.write(str_to_write)
        f.close()
        # generate new session Key
        generate_new_session_key()
    except Exception as e:
        logger.error(f"Error while writing new passwd in file: {e}", exc_info=True)
        raise e
