from typing import Dict, List, Tuple, Optional, Any, Mapping
from datetime import datetime, timezone
from zoneinfo import ZoneInfo
import calendar
import time
import ifaddr
import json
import logging
import os
import psutil
import shlex
import re
import subprocess
from ipaddress import ip_address, IPv4Address
from common_lib.lib.exceptions import SettingDateAndTimeError
from common_lib.lib.HTTPRequestsAPI import HTTPRequestsAPI



from common_lib.uci.UciCli import UciCli

from common_lib.lib.JsonUtils import read_from_json_file
from barix.system import get_cpu_temp

logger = logging.getLogger('flask-backend')

"""
def get_system_info(settings_requested: Mapping[str, str]) -> Dict[str, Any]:
    result_dict = {}
    try:
        for requestedSetting in settings_requested:
            if requestedSetting == "device_type":
                # get HW_type
                dev_type = get_device_type()
                result_dict["device_type"] = dev_type

            elif requestedSetting == "device_type_ID":
                # get HW_type_id
                dev_type_id = get_device_type_id()
                result_dict["device_type_ID"] = dev_type_id

            elif requestedSetting == "IPAM_type":
                ipam_type = get_ipam_type()
                result_dict["IPAM_type"] = ipam_type

            elif requestedSetting == "IPAM_type_ID":
                ipam_type_id = get_ipam_type_id()
                result_dict["IPAM_type_ID"] = ipam_type_id

            elif requestedSetting == "mac_addr":
                mac_addr = get_mac_addr()
                result_dict["mac_addr"] = mac_addr

            elif requestedSetting == "reg_id":
                reg_id = get_reg_id()
                result_dict["reg_id"] = reg_id

            elif requestedSetting == "ip_addr":
                ip_addr = get_ip_addr('eth0')
                result_dict["ip_addr"] = ip_addr

            elif requestedSetting == "netmask":
                netmask = get_netmask()
                result_dict["netmask"] = netmask

            elif requestedSetting == "default_gateway":
                default_gw = get_default_gw()
                result_dict["default_gateway"] = default_gw

            elif requestedSetting == "DNS_servers":
                dns_servers = get_dns_servers()
                result_dict["DNS_servers"] = dns_servers

            elif requestedSetting == "fw_version":
                # get fwVersion
                fw_version = get_fw_ver()
                result_dict["fw_version"] = fw_version

            elif requestedSetting == "rootFSDate":
                root_fs_date = get_root_fs_compilation_date()
                result_dict["rootFSDate"] = root_fs_date

            elif requestedSetting == "webUIVer":
                web_ui_ver = get_web_ui_version()
                result_dict["webUIVer"] = web_ui_ver

            elif requestedSetting == "app":
                result_dict["app"] = get_application()

            elif requestedSetting == "app_ver":
                # TODO: Duplicated. See "fw_version"
                result_dict["app_ver"] = get_fw_ver()

            elif requestedSetting == "kernelVer":
                kernel_ver = get_kernel_ver()
                result_dict["kernelVer"] = kernel_ver

            elif requestedSetting == "bootloaderVer":
                bootloader_ver = get_bootloader_ver()
                result_dict["bootloaderVer"] = bootloader_ver

            elif requestedSetting == "sys_time":
                result_dict["sys_time"] = get_system_time()

            elif requestedSetting == "local_time":
                result_dict["local_time"] = get_local_time()

            elif requestedSetting == "uptime":
                result_dict["uptime"] = get_up_time()

            elif requestedSetting == "licenses":
                result_dict['licenses'] = get_licenses()

            elif requestedSetting == "massStorageDev":
                result_dict["massStorageDev"] = get_mass_storage_devices()

            elif requestedSetting == "HW_type_ID":
                result_dict["HW_type_ID"] = get_hw_type_id()

            elif requestedSetting == "conn_status":
                result_dict["conn_status"] = get_wired_conn_status()

            elif requestedSetting == "poe_power_source":
                result_dict["poe_power_source"] = get_poe_config()

            elif requestedSetting == "poe_speaker_impedance":
                result_dict["poe_speaker_impedance"] = get_poe_spk_imp()

            elif requestedSetting == "poe_amp_gain":
                result_dict["poe_amp_gain"] = get_poe_amp_gain()

            elif requestedSetting == "otw_status":
                result_dict["otw_status"] = get_poe_over_temp_status()

            elif requestedSetting == "amp_temp":
                result_dict["amp_temp"] = get_amplifier_temperature()

            elif requestedSetting == "sensor_temp":
                result_dict["sensor_temp"] = get_temperature_from_sensor()

            # # @Deprecated
            # elif requestedSetting == "open_source_licenses":
            #     result_dict["open_source_licenses"] = get_open_source_licenses()

            elif requestedSetting == "open_source_info":
                result_dict["open_source_info"] = {"open_source_licenses": get_open_source_licenses(True),
                                                   "gpl_sources_url": get_gpl_sources_url()}

            elif requestedSetting == "storage_size":
                result_dict["storage_size"] = get_storage_size()

            elif requestedSetting == "network_settings":
                result_dict["network_settings"] = get_network_settings()

            elif requestedSetting == "cpu_temp":
                result_dict["cpu_temp"] = get_cpu_temp()

            elif requestedSetting == "defaults_content_enabled":
                result_dict["defaults_content_enabled"] = get_cpu_temp()

            elif requestedSetting == "capabilities":
                result_dict["capabilities"] = get_capabilities()
            else:
                logger.warning(f"Unrecognized setting {requestedSetting}")
    except Exception as e:
        logger.error(e, exc_info=True)
        raise e
    else:
        return result_dict
"""

def get_capabilities() -> Dict[str, bool]:
    return {
        # 'SNMPSettings': True if UciCli.get_value('snmp') else False,
        'SNMPSettings': False,
        'PoESettings': os.path.exists('/tmp/poe-detect'),
        'TempSensor': os.path.exists('/tmp/temp_sensor')
    }


def get_device_type() -> str:
    dev_type_file = "/tmp/factory_info_partition/factory_info_json"
    try:
        fact_info_json = read_from_json_file(dev_type_file)
        device_type = fact_info_json["HW_DEVICE"]["Product_Name"]
        return device_type
    except Exception as e:
        logger.error(f"Exception found: {e}", exc_info=True)
    return ""


def get_device_type_id() -> str:
    dev_type_file = "/tmp/factory_info_partition/factory_info_json"
    try:
        fact_info_json = read_from_json_file(dev_type_file)
        device_type_id = fact_info_json["HW_DEVICE"]["Product_ID"]
        return device_type_id
    except Exception as e:
        logger.error(f"Exception found: {e}", exc_info=True)
    return ""


def get_ipam_type() -> str:
    dev_type_file = "/tmp/factory_info_partition/factory_info_json"
    try:
        fact_info_json = read_from_json_file(dev_type_file)
        ipam_type = fact_info_json["HW_DEVICE"]["IPAM_Name"]
        return ipam_type
    except Exception as e:
        logger.error(f"Exception found: {e}", exc_info=True)
    return ""


def get_ipam_type_id() -> str:
    dev_type_file = "/tmp/factory_info_partition/factory_info_json"
    try:
        fact_info_json = read_from_json_file(dev_type_file)
        ipam_type_id = fact_info_json["HW_DEVICE"]["Legacy_IPAM_Type"]
        return ipam_type_id
    except Exception as e:
        logger.error(f"Exception found: {e}", exc_info=True)
    return ""


def get_mac_addr() -> str:
    mac_addr_file = "/sys/class/net/eth0/address"
    try:
        with open(mac_addr_file) as file:
            mac_addr = file.read().upper().strip('\n')
            return mac_addr
    except Exception as e:
        logger.error(f"Exception found: {e}", exc_info=True)
    return ""


def get_reg_id() -> str:
    config_file = "/mnt/shadow/config.json"
    try:
        config_json = read_from_json_file(config_file)
        reg_id = config_json["registrationId"]
        return reg_id
    except Exception as e:
        logger.error(f"Exception found: {e}", exc_info=True)
    return ""


def get_ip_addr(if_name: str = 'eth0') -> str:
    adapters = ifaddr.get_adapters()
    for adapter in adapters:
        if adapter.nice_name == if_name:
            for ip in adapter.ips:
                try:
                    if type(ip_address(str(ip.ip))) is IPv4Address:
                        return ip.ip
                except ValueError:
                    logger.warning(f"Invalid ip address for {if_name}")
    return ''


def get_netmask() -> str:
    process = subprocess.run(['ifconfig', 'eth0'], capture_output=True)
    output = process.stdout
    decoded_out = output.decode("utf-8")
    lines = decoded_out.split('\n')
    for line in lines:
        if line.find("netmask"):
            elems = line.split(' ')
            for idx in range(0, len(elems)):
                if elems[idx] == 'netmask':
                    return elems[idx + 1]
    return ''



# check /etc/network/interfaces
#
# auto eth0
# iface eth0 inet dhcp
#     dns-nameservers 8.8.8.8 1.1.1.1
#
def get_default_gw(if_name='eth0') -> str:
    """cmd = ["route", "-n", "|", "grep", "eth0", "|", "grep", "'^0.0.0.0'", "|", "awk", "'{print $2}'"]
    process = subprocess.run(cmd, capture_output=True)
    output = process.stdout
    print(output)
    return output
    """
    default_gw = None
    dhcp_enabled = UciCli.get_value(f'network.{if_name}.proto')
    if dhcp_enabled == "dhcp":
        # if DHCP is enabled, do this way
        ps1 = subprocess.Popen(['route', '-n'], stdout=subprocess.PIPE)
        ps2 = subprocess.Popen(['grep', if_name], stdin=ps1.stdout, stdout=subprocess.PIPE)
        ps3 = subprocess.Popen(['grep', '^0.0.0.0'], stdin=ps2.stdout, stdout=subprocess.PIPE)
        output = subprocess.check_output(['awk', '{print $2}'], stdin=ps3.stdout)
        ps1.wait()
        ps2.wait()
        ps3.wait()
        default_gw = output.decode("utf-8").strip('\n')
    elif dhcp_enabled == "static":
        # if DHCP is disabled, do this way
        default_gw = UciCli.get_value(f'network.{if_name}.gateway')
    return default_gw

def get_dns_servers() -> List[str]:
    file_path='/etc/resolv.conf'
    dns_servers = []
    try:
        with open(file_path, 'r') as file:
            for line in file:
                line = line.strip()
                if line.startswith('#') or not line:
                    continue
                parts = line.split()
                if parts[0] == 'nameserver' and len(parts) > 1:
                    dns_servers.append(parts[1])
    except Exception as e:
        logger.error(f"Error reading file: {e}")
    return dns_servers


def get_fw_ver() -> str:
    fw_ver_file = "/barix/info/VERSION"
    try:
        with open(fw_ver_file) as file:
            fw_version = file.read().strip('\n')
            return fw_version
    except Exception as e:
        logger.error(f"Exception found: {e}", exc_info=True)
    return ""


def get_application() -> str:
    return UciCli.get_value("flexa_app.AppParam.app_name")


def get_is_first_login() -> bool:
    """
        Checks if this is the user's first login.
        :returns: True if it is the first login, False otherwise.
    """
    return UciCli.get_value_as_boolean('httpd', 'webserver', 'first_login')


def set_is_first_login_false() -> bool:
    """
        Set user's first login to false.
    """
    UciCli.set_uci_configs({'httpd.webserver.first_login': False})


def get_kernel_ver() -> str:
    kernel_ver_file = "/proc/version"
    try:
        with open(kernel_ver_file) as file:
            kernel_ver = file.read().strip('\n')
            return kernel_ver
    except Exception as e:
        logger.error(f"Exception found: {e}", exc_info=True)
    return ""


def get_bootloader_ver() -> str:
    bootloader_ver_file = "/barix/info/BOOTVERSION"
    try:
        with open(bootloader_ver_file) as file:
            bootloader_ver = file.read().strip('\n')
            return bootloader_ver

    except Exception as e:
        logger.error(f"Exception found: {e}", exc_info=True)
    return ""


def get_system_time() -> float:
    """
    Get device UTC time
    """
    return time.time()


def get_local_time() -> Dict[str, Any]:
    """
    Get device local time
    """
    now = datetime.now()
    utc = now.astimezone(timezone.utc)
    time_zone = ZoneInfo(time.tzname[0])
    local = now.astimezone(time_zone)
    offset = local.strftime('%z')
    return {
        "utc": utc,
        "local": local,
        "time_zone": str(time_zone),
        "offset": offset
    }


def get_up_time() -> float:
    with open('/proc/uptime', 'r') as f:
        return float(f.readline().split()[0])


def get_licenses() -> Optional[List[Dict[str, str]]]:
    licenses = []
    try:
        process = subprocess.run(["qiba-spi-get-license.sh"], capture_output=True)
        output = process.stdout
        decoded_output = output.decode("utf-8")
        tmp_lic_list = decoded_output.split('\n')
        for tmp_lic in tmp_lic_list:
            if tmp_lic != '':
                data = tmp_lic.split(' ')
                return_dict = {"name": data[0], "status": data[5], "issueDate": data[3], "expireDate": data[4],
                               "id": data[1]}
                if data[2] == "N/A" or data[2] == "":
                    return_dict["features"] = data[2]
                else:
                    lic_features = data[2][5:-1]
                    return_dict["features"] = lic_features

                process = subprocess.run(["qiba-check-license", data[0]], capture_output=True)
                output = process.stdout

                decoded_output = output.decode("utf-8")

                if decoded_output.strip('\n') == data[2]:
                    return_dict["signature"] = "valid"
                else:
                    return_dict["signature"] = "invalid"

                licenses.append(return_dict)
    except Exception as e:
        logger.error(f"Error retrieving licenses from the device: {e}", exc_info=True)
        licenses = None
    return licenses


def get_mass_storage_devices() -> List[Dict[str, str]]:
    # {"mountedDevices":[{"type":"USB","mountAs":"rw","fsType":"vfat","size":"3.7G","used":"155.1M","available":"3.6G","use":"4%","mountPoint":"/run/media/sda1"}]}
    mass_storage_devices_list = []

    ps1 = subprocess.Popen(['df', '-h'], stdout=subprocess.PIPE)
    ps2 = subprocess.run(['grep', '/media/'], stdin=ps1.stdout, capture_output=True)
    ps1.wait()
    output = ps2.stdout
    storage_devices = output.decode("utf-8").split('\n')

    for line in storage_devices:
        if line != '':
            mass_storage_device_json = {}
            dev = line.split(' ')
            params_list = []
            for elem in dev:
                if elem != '':
                    params_list.append(elem)
            ps2 = subprocess.Popen(['mount'], stdout=subprocess.PIPE)
            output = subprocess.check_output(['grep', params_list[0]], stdin=ps2.stdout)
            ps2.wait()
            mount_line = output.decode("utf-8")
            mount_line_details = mount_line.split(' ')

            if mount_line_details[0].startswith("/dev/sd"):
                mass_storage_device_json["type"] = "USB"
            elif mount_line_details[0].startswith("/dev/mmcblk"):
                mass_storage_device_json["type"] = "microSD"
            else:
                return [{"error": "error"}]

            mass_storage_device_json["mountAs"] = mount_line_details[5].strip('(').split(',')[0]
            mass_storage_device_json["fsType"] = mount_line_details[4]
            mass_storage_device_json["size"] = params_list[1]
            mass_storage_device_json["used"] = params_list[2]
            mass_storage_device_json["available"] = params_list[3]
            mass_storage_device_json["use"] = params_list[4].strip('%')
            mass_storage_device_json["mountPoint"] = params_list[5]
            mass_storage_devices_list.append(mass_storage_device_json)

    return mass_storage_devices_list


def get_hw_type_id() -> str:
    dev_type_file = "/tmp/factory_info_partition/factory_info_json"
    try:
        fact_info_json = read_from_json_file(dev_type_file)
        hw_type_id = fact_info_json["HW_DEVICE"]["Legacy_HW_Type"]
        return hw_type_id
    except Exception as e:
        logger.error(f"Exception found: {e}", exc_info=True)
    return ""


def get_wired_conn_status() -> str:
    f = open("/sys/class/net/eth0/operstate", 'r')
    res = f.read()
    f.close()
    return res.strip('\n')


def map_poe_type_to_name(poe_type: str) -> str:
    if poe_type == "poe":
        return "PoE"
    elif poe_type == "poe+":
        return "PoE+"
    elif poe_type == "poe3":
        return "4PPoE"
    elif poe_type == "poe4":
        return "PoE Type 4"
    elif poe_type == "external":
        return "External"
    else:  # if uci does not exist, for example
        return "unknown"


def get_poe_config() -> str:
    poe_type = "not available"
    uci_poe_config = UciCli.get_value('poe.poe.config')
    if uci_poe_config == "auto":
        if os.path.exists('/tmp/poe-detect'):
            f = open("/tmp/poe-detect", "r")
            content = f.read()
            f.close()
            lines = content.split('\n')
            for line in lines:
                if line.startswith('poe.type'):
                    poe_type = line.split('=')[1]
                    break
            return map_poe_type_to_name(poe_type)
        else:
            return "not available"
    else:
        if uci_poe_config == '' or uci_poe_config is None:
            return "not available"
        return map_poe_type_to_name(uci_poe_config)


def get_poe_spk_imp() -> str:
    speaker_imp = "not available"
    uci_spk_imp = UciCli.get_value('speaker.speaker.config')
    if uci_spk_imp == "auto":
        """
        if os.path.exists('/tmp/speaker'):
            f = open("/tmp/speaker", "r")
            content = f.read()
            f.close()
            lines = content.split('\n')
            for line in lines:
                if line.startswith('speaker.impedance'):
                    speaker_imp = line.split('=')[1]
                    break
        else:
            speaker_imp = "not available"
        """
        try:
            r = HTTPRequestsAPI.get('http://localhost:50555/api/amp/speaker_imp')
            if r.status == 200:
                info = r.read()
                enc = r.info().get_content_charset('utf-8')
                json_out = json.loads(info.decode(enc))
                speaker_imp = json_out["impedance"]
        except Exception as e:
            logger.error(e, exc_info=True)

    elif uci_spk_imp is not None and uci_spk_imp != '':
        speaker_imp = uci_spk_imp

    return speaker_imp


def get_poe_amp_gain() -> str:
    amp_gain = "not available"
    try:
        r = HTTPRequestsAPI.get('http://localhost:50555/api/amp/gain')
        if r.status == 200:
            info = r.read()
            enc = r.info().get_content_charset('utf-8')
            json_out = json.loads(info.decode(enc))
            amp_gain = json_out["amp_gain"]
            if amp_gain == "0x30":
                amp_gain = "7.5V peak output voltage"
            elif amp_gain == "0x31":
                amp_gain = "15V peak output voltage"
            elif amp_gain == "0x32":
                amp_gain = "21V peak output voltage"
            elif amp_gain == "0x33":
                amp_gain = "29V peak output voltage"
            else:
                amp_gain = "unknown"
    except Exception as e:
        logger.error(e, exc_info=True)
    return amp_gain


def get_poe_over_temp_status() -> bool:
    try:
        r = HTTPRequestsAPI.get('http://localhost:50555/api/amp/over_temp')
        if r.status == 200:
            info = r.read()
            enc = r.info().get_content_charset('utf-8')
            json_out = json.loads(info.decode(enc))
            over_temp = json_out["over_temp"]
            return over_temp == 1
    except Exception as e:
        logger.error(e, exc_info=True)
    return False


def get_amplifier_temperature() -> str:
    try:
        r = HTTPRequestsAPI.get('http://localhost:50555/api/amp/temp')
        if r.status == 200:
            info = r.read()
            enc = r.info().get_content_charset('utf-8')
            json_out = json.loads(info.decode(enc))
            amp_temp = json_out["temperature"]
            return amp_temp
    except Exception as e:
        logger.error(e, exc_info=True)
    return ""


def get_temperature_from_sensor() -> str:
    if os.path.exists('/tmp/temp_sensor'):
        f = open("/tmp/temp_sensor", "r")
        content = f.read()
        f.close()
        return content
    return "not available"


def get_app_name_from_manifest() -> str:
    manifest_file_path = '/mnt/data/package/manifest.json'
    name = ''
    try:
        if os.path.isfile(manifest_file_path):
            data = read_from_json_file(manifest_file_path)
            if 'name' in data:
                name = data['name']
            else:
                name = "Flexa"
        else:
            name = "device"
    except Exception as e:
        logger.error(e, exc_info=True)
    return name


def get_root_fs_compilation_date() -> str:
    root_fs_date = None
    root_fs_date_file = "/barix/info/DATE"
    try:
        with open(root_fs_date_file) as file:
            root_fs_date = file.read().strip('\n')
    except Exception as e:
        logger.error(f"Error trying to get root_fs_date: {e}", exc_info=True)
    return root_fs_date


# TODO - check this
def get_web_ui_version() -> str:
    web_ui_ver = None
    web_ui_ver_file = "/usr/local/www/sys/VERSION"
    try:
        with open(web_ui_ver_file) as file:
            web_ui_ver = file.readline().strip('\n')
    except Exception as e:
        logger.error(f"Error trying to get webUiVersion: {e}", exc_info=True)
    return web_ui_ver


def get_device_alias() -> str:
    try:
        return UciCli.get_value('httpd', 'webserver', 'alias')
    except Exception as e:
        return ""

def get_log_lines(log_file_path, n_lines) -> str:
    lines = []
    string_lines = ""
    try:
        for line in reversed(list(open(log_file_path))):
            if n_lines > 0:
                lines.append(line)
                n_lines -= 1
            else:
                break
        for line in reversed(lines):
            string_lines += line
    except Exception as e:
        raise e
    else:
        return string_lines


def download_all_system_logs() -> str:
    app_name = get_app_name_from_manifest()
    mac_addr = get_mac_addr()
    timestamp = datetime.now().strftime("%Y%m%d%H%M%S")
    filename = app_name + "_" + mac_addr + "_" + timestamp + "_logs.tar.gz"
    filename = filename.replace(' ', '_')

    # remove any previous log.tar.gz that may exist
    try:
        my_dir = '/mnt/data/'  # enter the dir name
        for file_name in os.listdir(my_dir):
            if file_name.startswith((app_name + "_" + mac_addr + "_").replace(' ', '_')):
                os.remove(os.path.join(my_dir, file_name))
    except Exception as e:
        logger.error(e, exc_info=True)
        raise e

    proc = subprocess.run(['uci', 'show'], capture_output=True)
    uci_list_to_write = proc.stdout.decode("utf-8")
    try:
        f = open("/var/log/uci.config", "w")
        f.write(uci_list_to_write)
        f.close()
    except Exception as e:
        logger.error(f"Error creating /var/log/uci.config file: {e}", exc_info=True)
        raise e

    try:
        cmd = f"tar -czf /mnt/data/{filename} -C /var/log ."
        # Add interface captures files | IPAC-701
        interface_captures_dir = "/mnt/data/captures"
        if_cap_exist = os.path.exists(interface_captures_dir)
        if if_cap_exist:
            move_cmd = f"mv {interface_captures_dir} /var/log/"
            subprocess.run(shlex.split(move_cmd))
        # Tar all the content
        subprocess.run(shlex.split(cmd))
        # Remove interface captures dir | IPAC-701
        if if_cap_exist:
            rm_cmd = f"rm -rf /var/log/captures"
            subprocess.run(shlex.split(rm_cmd))
    except Exception as e:
        logger.error(e, exc_info=True)
        raise e

    return filename


def get_gpl_sources_url() -> str:
    content = None
    gpl_sources_url_file = "/barix/info/GPL_SOURCES_URL"
    try:
        with open(gpl_sources_url_file) as file:
            content = file.read()
    except Exception as e:
        logger.error(f"Error trying to get open source URL: {e}", exc_info=True)
    return content


def get_storage_size() -> dict[str, int]:
    try:
        storage_size_data: dict[str, int] = {'total': 0, 'used': 0, 'available': 0}
        ps1 = subprocess.Popen(['df'], stdout=subprocess.PIPE)
        ps2 = subprocess.run(['grep', '/mnt/data'], stdin=ps1.stdout, capture_output=True)
        ps1.wait()
        output = ps2.stdout
        storage_data = output.decode("utf-8").split('\n')

        for line in storage_data:
            if line != '':
                dev = line.split(' ')
                params_list = []
                for elem in dev:
                    if elem != '':
                        params_list.append(elem)

                if params_list[-1].strip('\n') == '/mnt/data':
                    # storage_size_data["total"] = int(params_list[1])
                    storage_size_data["used"] = int(params_list[2]) * 1024
                    storage_size_data["available"] = int(params_list[3]) * 1024
                    storage_size_data["total"] = storage_size_data["used"] + storage_size_data["available"]
                    # storage_size_data["use"] = int(params_list[4].strip('%'))
                    storage_size_data["use"] = round(storage_size_data["used"] * 100 / storage_size_data["total"])
        return storage_size_data
    except Exception as e:
        logger.error(f"Error getting storage size: {e}", exc_info=True)
        raise e


def get_network_settings() -> Dict[str, Dict]:
    network_settings = {}
    # get all network settings
    logger.debug(f"get_network_settings get adapters")
    network_adapters = ifaddr.get_adapters()
    for adapter in network_adapters:
        logger.debug(f"get_network_settings {adapter}")
        if adapter.name != 'lo':
            network_settings[adapter.name] = {}
            logger.debug(f"get_network_settings network_settings add {adapter.name}")
    for adapter in network_settings:
        logger.debug(f"get_network_settings adapter {adapter}")

        # get connection status of network adapters
        try:
            f = open("/sys/class/net/" + adapter + "/operstate", 'r')
            status = f.read().strip('\n')
            f.close()
        except Exception as e:
            logger.error(e, exc_info=True)
            status = "unknown"
            logger.debug(f"get_network_settings adapter {adapter} unknown")
        network_settings[adapter]["connection_status"] = status

        logger.debug(f"get_network_settings adapter {adapter} ifconfig")
        process = subprocess.run(['ifconfig', adapter], capture_output=True)
        output = process.stdout
        lines = output.decode("utf-8").split('\n')
        for line in lines:
            # get IP Addresses of network adapters
            if line.find("inet"):
                elems = line.split(' ')
                for idx in range(0, len(elems)):
                    if elems[idx] == 'inet':
                        network_settings[adapter]["ip_addr"] = elems[idx + 1]
            # get netmask of network adapters
            if line.find("netmask"):
                elems = line.split(' ')
                for idx in range(0, len(elems)):
                    if elems[idx] == 'netmask':
                        network_settings[adapter]["netmask"] = elems[idx + 1]
            # if "ip_addr" in network_settings[adapter].keys() and "netmask" in network_settings[adapter].keys():
            #    break

        logger.debug(f"get_network_settings adapter {adapter} route")
        # get default gateway of network adapters
        ps1 = subprocess.Popen(['route', '-n'], stdout=subprocess.PIPE)
        ps2 = subprocess.Popen(['grep', adapter], stdin=ps1.stdout, stdout=subprocess.PIPE)
        ps3 = subprocess.Popen(['grep', '^0.0.0.0'], stdin=ps2.stdout, stdout=subprocess.PIPE)
        ps1.wait()
        ps2.wait()
        ps3.wait()
        output = subprocess.check_output(['awk', '{print $2}'], stdin=ps3.stdout)
        network_settings[adapter]["default_gw"] = output.decode("utf-8").strip('\n')

        logger.debug(f"get_network_settings adapter {adapter} get_dns_servers")
        # get DNS servers
        #network_settings[adapter]["dns_servers"] = get_dns_servers(adapter)
        network_settings[adapter]["dns_servers"] = get_dns_servers()


    # logger.debug(f"network_settings \n{json.dumps(network_settings, indent=4)}")

    return network_settings


def set_date_and_time(string_date: str):
    """
    Set device's system Date and Time.
    @param string_date: string containing the date and time to set. Format must be: "YYYY-MM-DD HH:mm:ss"
    Note: When device reboots, date and time previously set may be lost.
    """
    try:
        cmd = "/bin/date -s \"" + string_date + "\""
        logger.info("Executing: {}".format(cmd))
        proc = subprocess.Popen(shlex.split(cmd), shell=False)
        proc.wait()
        if proc.returncode != 0:
            raise SettingDateAndTimeError
    except SettingDateAndTimeError:
        raise SettingDateAndTimeError
    except Exception as e:
        logger.error(e)


def _find_and_update_param(config_list, target_key, new_options, count = 0):
    """
    Recursively searches a list of config items for a param with target_key
    and updates its "options".
    """
    for item in config_list:
        c_type = item.get("c_type")
        c_data = item.get("c_data", {})

        if c_type == "param" and c_data.get("key") == target_key:

            type_props = c_data.get("type_properties", {})

            type_props["options"] = new_options

            c_data["type_properties"] = type_props
            item["c_data"] = c_data
            count +=1
            return True

        elif c_type == "section":
            content = c_data.get("content")
            if content and isinstance(content, list):
                # _find_and_update_param(content, target_key, new_options)
                if _find_and_update_param(content, target_key, new_options):
                    return True

    # return True if count else False
    return False
