import logging
from common_lib.lib.configuration import config
from common_lib.lib import syslog
from logging.handlers import RotatingFileHandler, SysLogHandler


def init_logger() -> logging.Logger:
    """
    Init logger
    Check from UCI if logger is local or remote
    """
    rem_syslog = syslog.SyslogHandle()
    if rem_syslog.is_enabled():
        init_syslog_server_logger(rem_syslog.get_server(), rem_syslog.get_port(), False, 'web')
    else:
        _init_local_logger(config['log_file_path'], config['log_file_max_size_mb'] * 1024 * 1024, config['log_file_count'], True)

    logger = logging.getLogger("flask-backend")
    logger.info("Starting web backend...")
    return logger


#implement a local rotating log with n log files maximum
def _init_local_logger(
    log_filename:str,
    max_file_size:int,
    num_backups:int,
    debug_mode:bool
):
    """
    Configure and initialize a logger with rotating file handler.

    Args:
        log_filename (str): Path to the log file.
        max_file_size (int): Maximum size of the log file before rotation (in bytes).
        num_backups (int): Number of backup log files to keep.
        debug_mode (bool, optional): If True, set log level to DEBUG; otherwise, INFO. Defaults to False.

    Returns:
        logging.Logger: Configured logger object.
    """
    rot_handler = RotatingFileHandler(log_filename, maxBytes=max_file_size, backupCount=num_backups)
    level_mode = logging.DEBUG if debug_mode else logging.INFO
    logging.basicConfig(level=level_mode,handlers=[rot_handler], format='%(asctime)s [%(levelname)-7s] %(filename)28s:%(lineno)3s | %(message)s')


class CustomSysLogHandler(SysLogHandler):
    """
    Use this class to be able to set a custom program name
    """
    def __init__(self, address, program_name, *args, **kwargs):
        super().__init__(address, *args, **kwargs)
        self.program_name = program_name

    def emit(self, record):
        msg = self.format(record)
        self.socket.sendto(f"<{self.facility | self.level}> {self.program_name}: {msg}".encode('utf-8'), self.address)


def init_syslog_server_logger(syslog_server_addr:str, syslog_server_port:int, debug_mode:bool, program_name:str):
    try:
        syslog_handler = CustomSysLogHandler(address=(syslog_server_addr, syslog_server_port), program_name=program_name)
        level_mode = logging.DEBUG if debug_mode else logging.INFO
        logging.basicConfig(level=level_mode,handlers=[syslog_handler], format='%(asctime)s [%(levelname)s] %(filename)s:%(lineno)s %(message)s')
    except Exception as e:
        print(f"Syslog Error: {e}")
