import logging
import os
import subprocess
import shlex
import time
import threading
import uuid
from common_lib.lib.configuration import config
from common_lib.lib.exceptions import InvalidUploadedFileError, FileTooLargeError

logger = logging.getLogger('flask-backend')

update_status = "clear"  # clear | uploading | installing | rebooting | error
errorStatus = None


def update_firmware(file_uploaded) -> None:
    """
    Starts a firmware update.
    @param file_uploaded: firmware update file uploaded

    - Raises an InvalidUploadedFileError if Form Data didn't upload the firmware update file using the "fwImage" key, or
        if filename is empty.
    - If file uploaded is too large, it will raise a FileTooLargeError Exception.
    - Checks if there's a FW update already running. If not exists a file to indicate that an FW update process has
        started (update_firmware_flag_file from config.json). This file is deleted when the process ends. Next time someone
        starts a FW update, if the file exists, it means that there's an update already running.
    - If there's a FW update already running, a FwUpdateRunningError Exception is raised.

    Then file is stored in the flask configured upload_files_dir and installed using the device's installer framework.
    When an Exception occurs and the process is interrupted, the flag that indicates that a firmware update is running
    (update_firmware_flag_file from config.json is deleted).
    """

    # Do before update
    global update_status
    logger.info("Starting Firmware Update")
    update_status = "uploading"

    # TODO: Call a callback to do stuff before update

    logger.info("Uploading new fw image...")
    create_update_firmware_flag()
    try:

        fw_update_file_path = config['upload_files_dir']
        os.makedirs(os.path.dirname(fw_update_file_path), exist_ok=True)

        if 'fwImage' not in file_uploaded:
            logger.error('fwImage is not in request files uploaded', exc_info=True)
            remove_update_firmware_flag()
            # remove file if error occurs
            raise InvalidUploadedFileError('Invalid file')

        uploaded_file = file_uploaded['fwImage']

        # if user does not select file, browser also
        # submit an empty part without filename
        if uploaded_file.filename == '':
            logger.error('filename from uploaded file is invalid', exc_info=True)
            remove_update_firmware_flag()
            # remove file if error occurs
            raise InvalidUploadedFileError('Invalid filename')


        # filename = secure_filename(uploaded_file.filename)
        filename = 'fw_update.tar'
        try:
            fw_updates_dir = config['upload_files_dir']
            logger.debug(f"Storing fw update file in {fw_updates_dir}")
            uploaded_file.save(os.path.join(fw_updates_dir, filename))
            t = threading.Thread(target=install_new_fw, args=(os.path.join(fw_updates_dir, filename),))
            t.start()

        except Exception as ex:
            remove_update_firmware_flag()
            update_status = "error"
            logger.error(ex, exc_info=True)
            # remove file if error occurs
            raise ex

    except InvalidUploadedFileError:
        remove_update_firmware_flag()
        raise InvalidUploadedFileError()

    except Exception as e:
        remove_update_firmware_flag()
        update_status = "error"
        logger.error(f"Error occurred while uploading fw file: {e}", exc_info=True)

        if str(e).startswith('413 Request Entity Too Large'):
            raise FileTooLargeError('REQUEST ENTITY TOO LARGE')
        else:
            raise e


def install_new_fw(file_path: str) -> None:
    """
    Install the fw update image.
    @param file_path: string containing the fullpath of the firmware update image
    If an exception occurs and the process is interrupted, the flag that indicates that an update is running is resetted (FW_UPDATE_FILE_PATH is deleted) and the fw update image is removed.
    If the firmware update finishes successfully, the fw update image is removed and the device reboots.
    """
    global update_status, errorStatus
    try:
        update_status = "installing"
        cmd = "qiba-update-client -f " + file_path
        logger.info(f"Installing new firmware <{file_path}>...")
        proc = None
        try:
            proc = subprocess.Popen(shlex.split(cmd), shell=False)
            proc.wait()

        except Exception as ex1:
            remove_update_firmware_flag()
            update_status = "error"
            logger.error(ex1, exc_info=True)
            # remove file if error occurs

        # log.debug("qiba-update-client returned {}".format(proc.returncode))
        if proc.returncode != 0:
            remove_update_firmware_flag()
            update_status = "error"
            logger.error("qiba-update-client failed installation", exc_info=True)
            errorStatus = proc.returncode

            # remove file
            try:
                os.remove(file_path)
            except Exception as ex2:
                logger.error(f"Can't remove file {file_path}: {ex2}", exc_info=True)
                # remove file if error occurs
        else:
            # remove file after installing it
            try:
                os.remove(file_path)
                update_status = "rebooting"
                time.sleep(3)
                logger.info("Rebooting...")
                subprocess.run('reboot')
            except Exception as ex3:
                update_status = "error"
                logger.error(f"Can't remove file {file_path}: {ex3}", exc_info=True)
                # remove file if error occurs
                remove_update_firmware_flag()

            # no need to removeUpdateFileChecker because device will reboot and /tmp will be cleaned
    except Exception as ex:
        logger.error(f"An error occurred: {ex}", exc_info=True)
    finally:
        remove_update_firmware_flag()
        pass


def create_update_firmware_flag() -> None:
    """
    Creates a flag that indicates that there's a FW update already running.
    If an error occurs, raise the Exception.
    """

    try:
        open(config['update_firmware_flag_file'], 'w').close()
    except Exception as ex:
        logger.error(f"An error occurred: {ex}", exc_info=True)
        raise ex


def remove_update_firmware_flag() -> None:
    """
    Resets the flag that indicates that there's a FW update already running.
    If an error occurs, raise the Exception.
    """

    update_firmware_flag_file = config['update_firmware_flag_file']

    try:
        if os.path.isfile(update_firmware_flag_file):
            os.remove(update_firmware_flag_file)
    except Exception as e:
        logger.error(f"Can't remove file {update_firmware_flag_file}: {e}", exc_info=True)
        raise e


def check_update_firmware_flag() -> bool:
    """
    Check if the update firmware is in progress.
    To check progress stage see "get_fw_update_status()"
    """
    return os.path.isfile(config['update_firmware_flag_file'])


def get_fw_update_status():
    """
    Gets a string containing the status of the FW update.
    It can be:
        - "clear" when nothing is happening or process has finished already;
        - "uploading" when the user is uploading a new FW update file image;
        - "installing" when the current state of the process is installing the FW update image;
        - "rebooting" when the firmware update is finished successfully and the device will reboot;
        - "error" if an error has occurred during the FW update process;
    """
    global update_status
    return update_status


def stop_fw_image_upload():
    """
    Abort the FW update process.
    Current status if updated to "error".
    Resets the flag that indicates that there's a FW update already running.
    """
    global update_status
    try:
        logger.info("Aborting Firmware Update")
        update_status = "error"
        remove_update_firmware_flag()
    except Exception as e:
        logger.error(e, exc_info=True)
        raise e


def get_erroget_error_status_message_status_message():
    """
    Gets a string containing the reason why the firmware update failed (if it failed).
    It can be:
        - 1	    Unable to get system information.
        - 5	    Failed extracting image.
        - 6	    Failed parsing image header file.
        - 8	    Version already installed.
        - 9	    Image checksum mismatch.
        - 10	Update does not match system machine.
        - 11	Update does not match system image.
        - 12	Signature validation failed.
        - 13	Update does not match system image.
        - 14	A newer version is in use.
        - 21	Installation failed.
    """
    if errorStatus == 1:
        return "Unable to get system information."
    elif errorStatus == 5:
        return "Failed extrating image."
    elif errorStatus == 6:
        return "Failed parsing image header file."
    elif errorStatus == 8:
        return "Version already installed."
    elif errorStatus == 9:
        return "Image checksum mismatch."
    elif errorStatus == 10:
        return "Update does not match system machine."
    elif errorStatus == 11:
        return "Update does not match system image."
    elif errorStatus == 12:
        return "Signature validation failed."
    elif errorStatus == 13:
        return "Update does not match system image."
    elif errorStatus == 14:
        return "A newer version is in use."
    elif errorStatus == 21:
        return "Installation failed."
    else:
        return errorStatus  # None
