from pathlib import Path
from typing import Optional

class EnvironmentChecker:
    """
    Utility class that reads the environment mode once and caches the result
    for all subsequent calls, avoiding multiple disk I/O operations.
    """

    # Define the constant, fixed, read-only system path
    SYSTEM_TYPE_PATH = "/barix/info/SYSTEM-TYPE"

    # Define the expected mode values
    MODE_DEVELOPMENT = "DEVELOPMENT"
    MODE_PRODUCTION = "PRODUCTION"

    def __init__(self):
        """
        Initializes the checker and sets up the internal cache.
        Note: The file is *not* read yet.
        """
        self._config_file_path = Path(self.SYSTEM_TYPE_PATH)
        self._cached_mode: Optional[str] = None # <-- The cache storage

    def _read_mode_from_file(self) -> Optional[str]:
        """
        Internal method to handle the actual file I/O.
        This is called only once by get_device_running_mode().
        """
        try:
            if not self._config_file_path.exists():
                print(f"Warning: System configuration file not found at {self._config_file_path}.")
                return None

            # READ operation
            content = self._config_file_path.read_text(encoding='utf-8').strip().upper()
            return content

        except IOError as e:
            print(f"Error reading configuration file {self._config_file_path}: {e}")
            return None


    def get_device_running_mode(self) -> Optional[str]:
        """
        Returns the environment mode, reading from the file only if not already cached.
        """
        # If the cache is empty (None), try to read the file
        if self._cached_mode is None:
            self._cached_mode = self._read_mode_from_file()

        # Return the cached value (which might be None if the file was missing/error)
        return self._cached_mode

    def is_device_running_develop_mode(self) -> bool:
        """
        Checks if the device is specifically running in development/debug mode.
        """
        current_mode = self.get_device_running_mode()

        # Check if the read mode matches the expected development constant
        return current_mode == self.MODE_DEVELOPMENT
