import logging
import os
import string
import random
import subprocess
import uuid
from typing import List
from OpenSSL import crypto
from pathlib import Path
from werkzeug.utils import secure_filename
from common_lib.lib.configuration import config
from common_lib.lib.exceptions import InvalidUploadedFileError
from common_lib.lib.system_info import get_storage_size

from common_lib.uci.UciCli import UciCli


logger = logging.getLogger('flask-backend')


def generate_ssl_certs(force=False):
    """
    Generate SSL Certificates to allow establishing an HTTPS connection with the WebUI
    """

    ssl_certificates_dir = config['ssl_certificates_dir']
    # if certs dir does not exist, create it
    p = Path(ssl_certificates_dir)
    if not p.exists():
        try:
            p.mkdir()
        except Exception as e:
            logger.error(e, exc_info=True)
            raise e

    # if certificate do not exist, generate it
    if not force and os.path.isfile(os.path.join(ssl_certificates_dir, 'barix.pem')):
        logger.debug("SSL certificate already exists, do not need to create it")
    else:
        try:
            # generate RSA key pair
            k = crypto.PKey()
            k.generate_key(crypto.TYPE_RSA, 2048)

            email_address = "info@barix.com"
            country_name = "CH"
            state_or_province_name = "Zurich"
            locality_name = "Dubendorf"
            organization_name = "Barix AG"
            common_name = "barix.local"
            cert = crypto.X509()
            cert.get_subject().C = country_name
            cert.get_subject().ST = state_or_province_name
            cert.get_subject().L = locality_name
            cert.get_subject().O = organization_name
            cert.get_subject().CN = common_name
            cert.get_subject().emailAddress = email_address
            cert.set_serial_number(uuid.uuid1().int)
            cert.gmtime_adj_notBefore(0)
            cert.gmtime_adj_notAfter(20 * 365 * 24 * 60 * 60)
            cert.set_issuer(cert.get_subject())
            cert.set_pubkey(k)
            cert.sign(k, 'sha256')

            with open(os.path.join(ssl_certificates_dir, 'barix.pem'), "w") as f:
                f.write(crypto.dump_privatekey(crypto.FILETYPE_PEM, k).decode("utf-8"))

            with open(os.path.join(ssl_certificates_dir, 'barix.pem'), "a") as f:
                f.write(crypto.dump_certificate(crypto.FILETYPE_PEM, cert).decode("utf-8"))

            # set chmod of certificate file
            os.chmod(os.path.join(ssl_certificates_dir, 'barix.pem'), 0o400)

            # set uci value
            # UciCli.set_uci_configs({'httpd.ssl.certificate':os.path.join(ssl_certificates_dir, 'barix.pem')}, restartServices=False)
            UciCli.set_uci_configs({'httpd.ssl.certificate': os.path.join(ssl_certificates_dir, 'barix.pem')})

        except Exception as e:
            logger.error(f"Error while generating SSL certificates: {e}", exc_info=True)
            raise e


def get_ca_certificates() -> List[dict]:
    """
    Get Custom Certificates installed on the device
    Returns a list of dictionaries, each representing a custom certificate.
    The information that each dictionary provides is:
        - name: name of the custom certificate
        - size: size of the custom certificate
        - state: state of the custom certificate ("Installed", ...)
    """
    try:
        certificates_list = []
        process = subprocess.run(['custom-ca-mgr', 'status'], capture_output=True)
        output = process.stdout
        decoded_output = output.decode("utf-8")
        lines = decoded_output.split('\n')
        for line in lines:
            elems = line.split(' ')
            if len(elems) > 1:  # last \n counts as one entire line
                # print(elems)
                filename = elems[1]
                # print(CA_CERTIFICATES_FOLDER)
                file_path = os.path.join(config['ssl_certificates_dir'], filename)
                # print(file_path)
                file_size = os.path.getsize(file_path)
                state = elems[0]
                certificates_list.append({"name": filename, "size": file_size, "state": state})
    except Exception as e:
        logger.error(e, exc_info=True)
        raise e
    else:
        logger.debug("Getting CA Certificates: {}".format(certificates_list))
        return certificates_list


def upload_ca_certificate(files_uploaded) -> str:
    """
    Upload a custom certificate
    @param files_uploaded:  file uploaded
    Raises an InvalidUploadedFileError if Form Data didn't upload the firmware update file using the "certificate" key, or if filename is empty.
    Saves the file temporarily in /tmp with a random name.
    Then checks if the file size is accepted and if device has space left.
    If file uploaded is too large, it will raise an InvalidUploadedFileError Exception too.
    It also validates if extension is valid. At the moment, the only certificate extension allowed is ".crt". If it isn't valid, raises an InvalidUploadedFileError.
    Finally, installs the certificate and moves the file from /tmp into the final destination: CA_CERTIFICATES_DIR
    If, during the process, an exception occurs, the file temporarily stored in /tmp is removed.
    """
    tmp_path = None
    try:
        if 'certificate' not in files_uploaded:
            raise InvalidUploadedFileError('Invalid file')
        uploaded_file = files_uploaded['certificate']
        # if user does not select file, browser also
        # submit an empty part without filename
        if uploaded_file.filename == '':
            raise InvalidUploadedFileError('Invalid filename')

        # generate random string
        letters = string.ascii_lowercase
        tmp_filename = ''.join(random.choice(letters) for i in range(8))
        tmp_path = os.path.join('/tmp', tmp_filename)

        uploaded_file.save(tmp_path)

        # check if file size is larger than available free space in storage (because of 5% root dedicated space, file is stored even if available space shows as 0, since backed is being run as root)
        storage_info = get_storage_size()
        available_space = storage_info["available"]
        if os.stat(tmp_path).st_size > available_space:
            raise Exception('No space left on device')

        # check file size
        filesize = os.stat(tmp_path).st_size
        if filesize < 133 or filesize > 16384000:
            raise InvalidUploadedFileError("File size is invalid. Please make sure you're uploading the right file.")

        # check file extension
        allowed_extensions = {'crt'}
        if '.' in uploaded_file.filename and uploaded_file.filename.rsplit('.', 1)[1].lower() in allowed_extensions:
            try:
                add_ca_certificates(tmp_path, secure_filename(uploaded_file.filename))
                return_msg = uploaded_file.filename + " installed."
                logger.info("Certificate " + return_msg)
                return return_msg
            except OSError:
                raise InvalidUploadedFileError('Invalid file.')
            except Exception as e:
                raise e
        else:
            raise InvalidUploadedFileError('Format not supported.')
    except Exception as e:
        if tmp_path is not None and os.path.isfile(tmp_path):
            os.remove(tmp_path)
        raise e


def add_ca_certificates(tmp_filepath: str, filename: str) -> None:
    """
    Install a custom certificate.
    @param tmp_filepath: string containing the full path of the certificate to install
    @param filename: string containing the name of the custom certificate
    Installs the certificate usinf openssl tool and, if successful, moves the file from its current directory into the final destination: CA_CERTIFICATES_DIR.
    custom-ca-mgr is also refreshed.
    If, during the process, some error occurs and the file has already been moved to its final destination, it is removed.
    """
    try:
        # print('openssl x509 -in /tmp/foo')
        process = subprocess.run(['openssl', 'x509', '-in', tmp_filepath], capture_output=True)
        stderr = process.stderr
        empty = ''.encode("utf-8")
        if stderr != empty:
            raise OSError

    except Exception as e:
        logger.error(e, exc_info=True)
        raise e

    final_filepath = None
    try:
        final_filepath = os.path.join(config['ssl_certificates_dir'], filename)
        os.makedirs(os.path.dirname(final_filepath), exist_ok=True)
        subprocess.run(['mv', tmp_filepath, final_filepath])
        # print('custom-ca-mgr', 'refresh')
        logger.info("Certificate {} installed successfully.".format(filename))
        subprocess.run(['custom-ca-mgr', 'refresh'])

    except Exception as e:
        logger.error(e, exc_info=True)
        if final_filepath is not None and os.path.isfile(final_filepath):
            os.remove(final_filepath)
        raise e


def remove_ca_certificates(certificates_list: List[str]) -> int:
    """
    Remove custom certificates
    @param certificates_list: list of certificates names to remove
    Returns the number of files that were successfully removed.
    For each name of the list, is there's a file in the CA_CERTIFICATES_DIR that matches the name, the file is removed.
    At the end, custom-ca-mgr is refreshed.
    """
    try:
        counter = 0
        for certificateName in certificates_list:
            file_path = os.path.join(config['ssl_certificates_dir'], certificateName)
            if os.path.isfile(file_path):
                os.remove(file_path)
                logger.info(f"Certificate {file_path} removed")
                counter += 1
        subprocess.run(['custom-ca-mgr', 'refresh'], capture_output=False)
        return counter

    except Exception as e:
        logger.error(e, exc_info=True)
        raise e


def certificate_is_valid(cert_path: str) -> bool:
    try:
        # process = subprocess.run(['openssl', 'x509', '-in', cert_path, '-noout'])
        process = subprocess.run(['openssl', 'verify', '-CAfile', cert_path, cert_path])
        if process.returncode == 0:
            return True
        else:
            return False
    except Exception as e:
        logger.error(e, exc_info=True)
        raise e


def key_is_valid(key_path: str) -> bool:
    try:
        process = subprocess.run(['openssl', 'rsa', '-in', key_path, '-check', '-noout'])
        if process.returncode == 0:
            return True
        else:
            return False
    except Exception as e:
        logger.error(e, exc_info=True)
        raise e
