from typing import List, Set, Union, Any
import logging
import os
import subprocess
import time
from common_lib.lib.configuration import config

from common_lib.uci.Uci2ServiceMap import Uci2ServiceMap

logger = logging.getLogger('flask-backend')


class RestartServiceUtils:
    """
    Reads the file RESTART_SERVICES_FILE_PATH containing a list of services to restart and restarts them by order of priority.
    The list should contain one string per line and the function is going to iterate by each line until there is no more lines to read.
    When a service string is restarted, it is removed from the list. After each iteration, the list is rewritten in the file.
    If there is a string "reboot" in the file, the device reboots and the loop is finished without reading any other line, because when the device reboots,
    all services will be restarted by order and RESTART_SERVICES_FILE_PATH is removed, since it is stored in /tmp.
    Then checks if the backend needs to restart. When the backends are launched, it will check the list again to see if there are services left to restart.
    After that, it will sort the services by priority (each service has a SXX prefix: check /etc/rc5.d) and restart them one by one.
    """

    '''
    Reads the file RESTART_SERVICES_FILE_PATH containing a list of services to restart.
    Returns an empty list if the file doesn't exist or a list with the content of the file, where each element corresponds to a line of the file.
    '''

    @staticmethod
    def read_from_restart_services_file() -> List[str] or None:
        services_from_file = []
        try:
            if os.path.isfile(config['restart_services_file_path']):
                fd = open(config['restart_services_file_path'], 'r')
                content = fd.readlines()
                for elem in content:
                    services_from_file.append(elem.strip('\n'))
        except Exception as e:
            logger.error(f"An error occurred while reading from file {config['restart_services_file_path']}:{e}", exc_info=True)
            raise e
        else:
            return services_from_file

    '''
    Writes the provided list of services to restart to the RESTART_SERVICES_FILE_PATH file.
    @param listOfServices: list of services to restart
    Each list element will correspond to a line in the file. Lines are separated by \n character.
    '''

    @staticmethod
    def write_into_restart_services_file(services_list: Set[Union[str, Any]] or List) -> None:
        try:
            str_services_list = ""
            for service in services_list:
                str_services_list = str_services_list + service + "\n"
            str_services_list = str_services_list[:-1]  # remove last \n
            fd = open(config['restart_services_file_path'], 'w')
            fd.write(str_services_list)
            fd.close()
        except Exception as e:
            logger.error(f"Error writing into {config['restart_services_file_path']}:{e}", exc_info=True)
            raise e

    @staticmethod
    def restart_services_from_file() -> None:
        if not os.path.isfile(config['restart_services_file_path']):
            logger.info(f"No services to restart: file {config['restart_services_file_path']} does not exist")

        try:
            # read services from file
            services_list = RestartServiceUtils.read_from_restart_services_file()

            if "reboot" in services_list:
                logger.debug("run command: reboot")
                subprocess.run("reboot")
                time.sleep(1)
                return
            
            services_list = list(set(services_list))  # Remove duplicated services
            
            if "webui" in services_list:
                commands = Uci2ServiceMap().get_service_command(services_list[services_list.index('webui')])

                services_list.remove('webui')
                RestartServiceUtils.write_into_restart_services_file(services_list)

                logger.debug(f"restarting webui - commands: {commands}")
                for command in commands:
                    subprocess.run(command, shell=True)
                time.sleep(1)
            
            sorted_services = Uci2ServiceMap().sort_services_by_priority(services_list)

            for service in sorted_services:
                commands = Uci2ServiceMap().get_service_command(service)
                logger.debug(f"restarting {service} - commands: {commands}")
                for command in commands:
                    subprocess.run(command, shell=True)
                    time.sleep(1)  # is this required?

                # time.sleep(1)  # is this required?
            with open(config['restart_services_file_path'], 'w') as f:
                f.write("")

        except Exception as e:
            logger.error(f"An error occurred while restarting services: {e}", exc_info=True)
            raise e

