import json
import logging

log = logging.getLogger('flask-backend')


def read_from_json_file(file_path: str) -> dict:
    """
    Reads from a JSON file provided.
    - If the file doesn't exist, raises a FileNotFoundError.
    - If the file exists but is empty, raises a JSONDecodeError.
    - If the file contains invalid JSON, raises a JSONDecodeError.

    @param file_path: string containing the full path of the JSON file to read
    @return: dictionary containing the parsed JSON data
    """
    try:
        with open(file_path, 'r') as file:
            # Attempt to load JSON content
            return json.load(file)
    except FileNotFoundError:
        log.error(f"File not found: {file_path}", exc_info=True)
        raise  # Re-raise FileNotFoundError
    except json.JSONDecodeError:
        log.warning(f"File {file_path} is empty or contains invalid JSON. Returning an empty dictionary.")
        raise  # Re-raise empty dictionary for invalid or empty files
    except Exception as e:
        log.error(f"Unexpected error reading JSON file {file_path}: {e}", exc_info=True)
        raise  # Re-raise any other unexpected exceptions
