import json
import logging
import re

logger = logging.getLogger('flask-backend')



class JsonRules:

    @staticmethod
    def get_value(config, section, option, data, default):
        try:
            return data[config][section][option]
        except:
            return default


    @staticmethod
    def convert_to_types(data, types):
        def to_boolean(value):
            if isinstance(value, bool):
                return value
            if isinstance(value, str):
                return value.lower() in ('true', '1', 'yes', 'on', 't', 'y')
            if isinstance(value, (int, float)):
                return bool(value)
            raise f"could't convert {value} to boolean"

        def convert_value(value, target_type):
            # Convert the value to the desired type
            try:
                if target_type == "int":
                    return int(value)
                elif target_type == "float":
                    return float(value)
                elif target_type == "boolean":
                    return to_boolean(value)
                elif target_type == "json":
                    return json.loads()  # Convert string to valid JSON
                else:
                    return str(value)  # Default type
            except (ValueError, json.JSONDecodeError):
                return value  # Return original value if conversion fails

        def recursive_convert(data, types):

            if '__type__' in types:
                types = types['__type__']

            # Recursively traverse both JSON structures
            if isinstance(data, dict) and isinstance(types, dict):
                for key in data.keys():
                    if key in types:
                        data[key] = recursive_convert(data[key], types[key])
                    else:
                        # data[key] = recursive_convert(data[key], "str")  # Default type
                        data[key] = recursive_convert(data[key], {})  # Default type
            elif isinstance(data, list):
                return [convert_value(item, types) for item in data]  # Handle lists
            elif isinstance(types, str):
                return convert_value(data, types)  # Base case for conversion
            return data

        return recursive_convert(data, types)


    @staticmethod
    def json_to_types(ucis, mappingFilePath):
        settings = {}
        with open(mappingFilePath) as f:
            try:
                settings = json.load(f)
            except Exception as e:
                logger.debug(f"Couldn't load json information, error: {e}")
                raise e

        data = JsonRules.convert_to_types(ucis, settings)
        return data


    @staticmethod
    def validate_rules(data, rules, path=""):
        errors = []

        for key, rules in rules.items():
            current_path = f"{path}.{key}" if path else key

            # Check for required fields
            # if isinstance(rules, dict) and rules.get("required", False) and key not in data:
            #     errors.append(f"Missing required key: {current_path}")
            #     continue

            # If the key exists, validate the value
            if key in data:
                value = data[key]

                if isinstance(rules, dict) and "__type__" in rules:
                    expected_type = rules["__type__"]

                    # Type validation
                    if expected_type == "int" and not isinstance(value, int):
                        errors.append(f"Invalid type for {current_path}: Expected int, got {type(value).__name__}")
                    elif expected_type == "str" and not isinstance(value, str):
                        errors.append(f"Invalid type for {current_path}: Expected str, got {type(value).__name__}")
                    elif expected_type == "boolean" and not isinstance(value, bool):
                        errors.append(f"Invalid type for {current_path}: Expected boolean, got {type(value).__name__}")

                    # Limits validation
                    if "limits" in rules and isinstance(value, int):
                        min_val, max_val = rules["limits"]
                        if not (min_val <= value <= max_val):
                            errors.append(f"Value out of range for {current_path}: Expected between {min_val} and {max_val}, got {value}")

                    # Options validation
                    if "options" in rules and isinstance(value, str):
                        if value not in rules["options"]:
                            errors.append(f"Invalid option for {current_path}: Expected one of {rules['options']}, got {value}")

                    if "regex" in rules and isinstance(value, str):
                        regex=re.compile('^[a-z][a-z, ]*$')
                        if not regex.match(value):
                            errors.append(f"Invalid format for {current_path}")

                elif isinstance(rules, dict):  # Nested dictionary case
                    if isinstance(value, dict):
                        errors.extend(JsonRules.validate_rules(value, rules, current_path))
                    else:
                        errors.append(f"Invalid structure for {current_path}: Expected a nested object")

        return errors
