from pathlib import Path
import json

class FileUtils:
    @staticmethod
    def get_file(filepath, logger, mode='r', encoding='utf-8'):
        """
        Reads and returns the content of a file.

        :param filepath: Path to the file (string or Path object)
        :param logger: Logger instance for error reporting
        :param mode: File open mode ('r' for text, 'rb' for binary)
        :param encoding: Encoding to use when reading text files
        :return: File content as string/bytes, or None if error occurs
        """
        path_obj = Path(filepath)

        # Check existence
        if not path_obj.exists():
            logger.error(f"File not found: {path_obj}")
            return None

        # Check it's actually a file
        if not path_obj.is_file():
            logger.error(f"Path is not a file: {path_obj}")
            return None

        # Try to read
        try:
            if 'b' in mode:
                with path_obj.open(mode) as file:
                    return file.read()
            else:
                with path_obj.open(mode, encoding=encoding) as file:
                    return file.read()
        except Exception as e:
            logger.error(f"Error reading file {path_obj}: {e}", exc_info=True)
            return None


    @staticmethod
    def get_json(filepath, logger, encoding='utf-8'):
        """
        Loads and returns a dictionary from a JSON file.

        :param filepath: Path to the file
        :param logger: Logger instance
        :param encoding: File encoding
        :return: A dictionary, or None if error occurs
        """
        path_obj = Path(filepath)

        if not path_obj.exists() or not path_obj.is_file():
            logger.error(f"File not found or is not a file: {path_obj}")
            return None

        try:
            with path_obj.open('r', encoding=encoding) as file:
                return json.load(file)
        except json.JSONDecodeError as e:
            logger.error(f"Error parsing JSON from file {path_obj}: {e}")
            return None
        except Exception as e:
            logger.error(f"Error reading file {path_obj}: {e}")
            return None
