import os
from flask import Flask
from datetime import timedelta
from pathlib import Path
from common_lib.lib.MemoryRequest import MemoryRequest
from common_lib.lib.RestartServiceUtils import RestartServiceUtils
from common_lib.lib.cryptography import get_device_identifier
from common_lib.lib.configuration import config

from routes import include_static_routes, include_web_api_routes



class AppUtils:

    @staticmethod
    def set_base_configuration(app, logger) -> Flask:
        """
        Configure app
        :param app:
        :param logger:
        :return:
        """

        # Check if it has services to restart
        RestartServiceUtils.restart_services_from_file()

        AppUtils.grant_upload_dir(logger)

        app.secret_key = get_device_identifier()
        app.config['SESSION_TYPE'] = 'memcached'  # Other options include 'redis', 'filesystem', etc.
        app.config['PERMANENT_SESSION_LIFETIME'] = timedelta(days=365)

        # UPLOAD_FOLDER is where uploaded files are stored
        app.config['UPLOAD_FOLDER'] = config['upload_files_dir']

        # maximum size allowed for the files to upload
        app.config['MAX_CONTENT_LENGTH'] = config['max_file_size_upload']

        # override default werkzeug form parser to allow to update files to the UPLOAD_FOLDER instead of default
        app.request_class = MemoryRequest

        # Add base routes
        include_static_routes(app)
        include_web_api_routes(app)
        return app

    @staticmethod
    def run_app(app, logger):
        """
        Get web protocol from uci to launch application
        """

        # nginx deals with (http/https)
        try:
            app.run(host='0.0.0.0', port=80, threaded=True)
        except Exception as e:
            logger.error(e, exc_info=True)

    @staticmethod
    def grant_upload_dir(logger):
        """
        Check if needed directories exists
        """
        upload_files_path = Path(config['upload_files_dir'])
        if not upload_files_path.exists():
            try:
                upload_files_path.mkdir()
            except Exception as e:
                logger.error("Error creating upload files directory {}: {}".format(config['upload_files_dir'], e), exc_info=True)

        config_dir_path = Path(config['config_dir'])
        if not config_dir_path.exists():
            try:
                config_dir_path.mkdir()
            except Exception as e:
                logger.error("Error creating config directory {}: {}".format(config['config_dir'], e), exc_info=True)
