import os
import json
import logging
import subprocess
import re
import tempfile
import shutil
import urllib.parse
from datetime import datetime

from os import walk

# from typing import Dict, List, Tuple, Optional, Any, Mapping

from flask import Response, request, jsonify, send_file, current_app, after_this_request, make_response
# from flask import Response
from werkzeug.utils import secure_filename
from application.application_routes import application_bp

from common_lib.uci.Utils import Utils
from common_lib.uci.UciCli import UciCli
from common_lib.rest.web.midleware.login_required import login_required
from common_lib.rest.web.midleware.update_state_validator import update_state_validator

from barix.web.uci import getValueOfUci, setUciConfigs

from uci import UciException, UciExceptionNotFound

from common_lib.lib.system_info import \
    get_mac_addr, \
    get_hw_type_id, \
    get_device_type, \
    get_device_type_id, \
    get_ipam_type, \
    get_ipam_type_id, \
    get_kernel_ver, \
    get_local_time, \
    get_capabilities, \
    get_network_settings, \
    get_up_time, \
    get_licenses, \
    get_mass_storage_devices, \
    get_log_lines, \
    get_fw_ver, \
    get_device_alias

from common_lib.lib.JsonRules import JsonRules
from common_lib.lib.FileUtils import FileUtils

from application.local_lib import \
    uci_to_json


import paho.mqtt.client as mqtt

# MQTT Broker configuration
BROKER = "127.0.0.1"
PORT = 1883
TOPIC_BASE = "RTP"


logger = logging.getLogger('flask-backend')

try:
    product_info = FileUtils.get_json('config/product-info.json', logger)
    DEVICE_NAME = product_info.get("device_name", "Barix Device")
    logger.info(f"Successfully loaded device name: {DEVICE_NAME}")
except Exception as e:
    logger.error(f"Failed to load product info from 'product-info.json': {e}")
    DEVICE_NAME = "Barix Device"


def _run_cmd_to_file(cmd_list, file_path):
    """Helper to run a shell command and save stdout to a file."""
    try:
        with open(file_path, "w") as f:
            subprocess.run(cmd_list, stdout=f, stderr=subprocess.PIPE, check=False)
    except Exception as e:
        logger.error(f"Failed to run command {' '.join(cmd_list)}: {e}")


@application_bp.route('/system/status', methods=['GET'])
@login_required
@update_state_validator
def application_get_system_status():
    try:
        # Existing logic for stream settings
        portal_url = UciCli.get_value("application","main_config","backend_url")
        active_stream = UciCli.get_value("application","main_config","active_url")
        priority_stream = UciCli.get_value("application","main_config","priority_url")
        main_stream = UciCli.get_value("application","main_config","playback_url")
        failover_stream = UciCli.get_value("application","main_config","failover_url")
        playback_volume = UciCli.get_value("application","audio","player_volume_percent")

        # 1. Retrieve the base network settings
        network_data = get_network_settings()

        # 2. Manually fetch wlan0 settings since get_network_settings misses it
        try:
            # Fetch the raw network config
            uci_network_raw = UciCli.get_value("network")

            # Check if wlan0 exists in the uci config
            if uci_network_raw and "wlan0" in uci_network_raw:
                # Wrap it to match the structure JsonRules expects
                uci_network_wrapper = { "network": uci_network_raw }
                key = "wlan0"

                interface = {
                    "protocol":     JsonRules.get_value("network", key, "proto",     uci_network_wrapper, "dhcp"),
                    "ip_address":       JsonRules.get_value("network", key, "ipaddr",    uci_network_wrapper, ''),
                    "netmask":      JsonRules.get_value("network", key, "netmask",   uci_network_wrapper, ''),
                }

                if interface["protocol"]:
                    interface["protocol"] = interface["protocol"].lower()

                    try:
                        cmd = ['/sbin/barix-wifi-status', '--json', 'wlan0']

                        result = subprocess.run(cmd,
                                capture_output=True,
                                text=True,
                                check=True,
                                encoding='utf-8')

                        status_data = json.loads(result.stdout)
                        interface["connection_status"] = "up"
                        interface["mac_address"] = status_data["mac_address"]
                    except Exception as e:
                        interface["connection_status"] = "down"
                        interface["mac_address"] = "N/A"

                # Extract wlan0 specific properties
                interface["ssid"] = UciCli.get_value("network", key, "ssid")
                interface["security"] = UciCli.get_value("network", key, "security")

                # Add wlan0 to the network_data dictionary
                network_data[key] = interface

        except Exception as e:
            # Log error but don't fail the whole request if wlan0 fetch fails
            logger.error(f"Failed to append wlan0 settings: {e}")

        data = {
            "application": {
                "portal_url": portal_url,
                "active_stream": active_stream,
                "priority_stream": priority_stream,
                "main_stream": main_stream,
                "failover_stream": failover_stream,
                "playback_volume": playback_volume
            },
            "device": {
                "Mac_addr":         get_mac_addr(),
                "HW_type_ID":       get_hw_type_id(),
                "type":             get_device_type(),
                "type_ID":          get_device_type_id(),
                "IPAM_type":        get_ipam_type(),
                "IPAM_type_ID":     get_ipam_type_id(),
                "kernelVer":        get_kernel_ver(),
                "local_time":       get_local_time(),
                "uptime":           get_up_time(),
                "fw_version":       get_fw_ver(),
                "alias":            get_device_alias(),
                "device_name":      DEVICE_NAME
            },
            "capabilities": get_capabilities(),
            "network":      network_data,
            "licenses":     get_licenses(),
            "massStorageDev": get_mass_storage_devices()
        }

        return jsonify(data), 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return jsonify(str(e)), 500


@application_bp.route('/system/status/config', methods=['GET'])
@login_required
@update_state_validator
def get_system_status_config():
    filepath = '/barix/apps/webui/application/sdf/system-status-config.json'
    try:
        with open(filepath, "r") as f:
            config = json.load(f)
    except Exception as e:
        logger.error(f'Error reading SDF config file "{filepath}": {e}', exc_info=True)

    if not config:
        return '', 500

    return jsonify(config), 200

@application_bp.route('/system/status/wifi', methods=['GET'])
@login_required
@update_state_validator
def get_system_wifi_status():
    filepath = '/tmp/barix-wifi/scan-results.json'
    results = None

    try:
        with open(filepath, "r") as f:
            content = f.read().strip()
            if content:
                results = json.loads(content)
            else:
                results = {}
    except FileNotFoundError:
        logger.error(f'WiFi scan results file not found: "{filepath}"')
        return '', 404
    except json.JSONDecodeError as e:
        logger.error(f'Invalid JSON in "{filepath}": {e}', exc_info=True)
        return '', 500
    except Exception as e:
        logger.error(f'Error reading wifi scan results file "{filepath}": {e}', exc_info=True)
        return '', 500

    try:
        cmd = ['/sbin/barix-wifi-status', '--json', 'wlan0']

        result = subprocess.run(cmd,
                                capture_output=True,
                                text=True,
                                check=True,
                                encoding='utf-8')

        status_data = json.loads(result.stdout)

        results['status'] = status_data

    except (FileNotFoundError, subprocess.CalledProcessError) as e:
        logger.error(f'Error executing "{e.cmd}": {e}')
        results['status'] = {'error': 'Could not retrieve WiFi status', 'details': str(e)}
    except json.JSONDecodeError as e:
        logger.error(f'Failed to decode JSON from "{" ".join(cmd)}": {e}')
        results['status'] = {'error': 'Invalid status data from WiFi command'}
    except Exception as e:
        logger.error(f'Unexpected error getting WiFi status: {e}', exc_info=True)
        results['status'] = {'error': 'Unknown error retrieving WiFi status'}

    return jsonify(results), 200




@application_bp.route('/settings/basic', methods=['GET'])
@login_required
@update_state_validator
def application_get_basic_settings():
    try:
        data_device = {}
        data_device['alias'] = get_device_alias()

        # with open("/barix/config/current/network", 'r') as f:
        #     uci_data = f.readlines()

        priority_url = UciCli.get_value("application","main_config","priority_url")

        rtp_out_enable = UciCli.get_value("audioplex","rtp_out","enable")
        rtp_out_url = UciCli.get_value("audioplex","rtp_out","url")
        rtp_out_audio_format = UciCli.get_value("audioplex","rtp_out","audio_format")

        uci_network = { "network": UciCli.get_value("network")}
        sonic_ip = UciCli.get_value("sonic_ip","general")

        data_ntp = UciCli.get_value("ntp", "source")
        ssh_enabled = UciCli.get_value("dropbear", "RunCtl", "enable")

        web_protocol = UciCli.get_value("httpd", "webserver", "protocol")

        data_network = {}

        proxy_config = {
            "enable": UciCli.get_value("proxy_config", web_protocol, "enable"),
            "host": UciCli.get_value("proxy_config", web_protocol, "host"),
            "port": UciCli.get_value("proxy_config", web_protocol, "port"),
            "username": UciCli.get_value("proxy_config", web_protocol, "username"),
            "noproxy": UciCli.get_value("proxy_config", web_protocol, "no_proxy"),
        }

        data_network["proxy_config"] = proxy_config

        for key in uci_network["network"]:
            interface={
                "protocol":     JsonRules.get_value("network", key, "proto",     uci_network, "dhcp"),
                "ipaddr":    JsonRules.get_value("network", key, "ipaddr",    uci_network, ''),
                "netmask":   JsonRules.get_value("network", key, "netmask",   uci_network, ''),
                "gateway":   JsonRules.get_value("network", key, "gateway",   uci_network, ''),
                "dhcpname":  JsonRules.get_value("network", key, "dhcpname",  uci_network, ''),
                "dns_type":  JsonRules.get_value("network", key, "dns_type",  uci_network, "AUTO"),
                "dns1":      JsonRules.get_value("network", key, "dns1",      uci_network, ''),
                "dns2":      JsonRules.get_value("network", key, "dns2",      uci_network, '')
            }
            interface["protocol"]    = interface["protocol"].lower()
            interface["dns_type"] = interface["dns_type"].lower()

            if key == "wlan0":
                interface["ssid"] = UciCli.get_value("network", key, "ssid")
                interface["security"] = UciCli.get_value("network", key, "security")
                # interface["psk"] = UciCli.get_value("network", key, "psk")

            data_network[key]=interface

        data_device["sonic_ip"] = sonic_ip

        data_network["protocol"] = web_protocol


        json_data = {
            "application" : {
                "priority_url": priority_url,
                "rtp_out": {
                    "enable": rtp_out_enable,
                    "url": rtp_out_url,
                    "audio_format": rtp_out_audio_format
                }
            },
            "device" :  data_device,
            "network" : data_network,
            "time_settings" : {
                "ntp":      data_ntp,
            },
            "security": {
                "ssh" : ssh_enabled,
            }
        }

        json_converted = JsonRules.json_to_types(json_data, "./application/rules/basic-settings.rules.json")

        logger.info("Loaded basic settings values")
        return jsonify(json_converted), 200
    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500


@application_bp.route('/settings/basic', methods=['POST'])
@login_required
@update_state_validator
def application_set_basic_settings():
    try:
        list_ucis = {}
        data = json.loads(request.data)

        #
        # validate_rules
        #
        with open("./application/rules/basic-settings.rules.json", 'r') as f:
            data_rules = json.load(f)
        validation_errors = JsonRules.validate_rules(data, data_rules)
        if validation_errors:
            retval = {
                "message": "Validation errors found",
                "errors": validation_errors
            }
            return jsonify(retval), 400

        if 'application' in data:
            app_data = data['application']

        if 'priority_url' in app_data:
            list_ucis['application.main_config.priority_url'] = app_data['priority_url']

        if 'rtp_out' in app_data:
            rtp_data = app_data['rtp_out']
            if 'enable' in rtp_data:
                list_ucis['audioplex.rtp_out.enable'] = rtp_data['enable']
            if 'url' in rtp_data:
                list_ucis['audioplex.rtp_out.url'] = rtp_data['url']
            if 'audio_format' in rtp_data:
                list_ucis['audioplex.rtp_out.audio_format'] = rtp_data['audio_format']

        if 'device' in data and 'alias' in data["device"]:
            logger.debug(f"Alias: {data['device']['alias']}")
            list_ucis["httpd.webserver.alias"] = data["device"]["alias"]

        if 'time_settings' in data:
            if 'ntp' in data['time_settings']:
                try:
                    list_ucis["ntp.source.server1"] = data['time_settings']['ntp']['server1']
                except:
                    return f"Missing NTP server 1 parameter", 400
                list_ucis["ntp.source.server2"] = data['time_settings']['ntp']['server2']
                list_ucis["ntp.source.server3"] = data['time_settings']['ntp']['server3']

        if 'security' in data and 'ssh' in data['security']:
            list_ucis["dropbear.RunCtl.enable"] = data['security']['ssh']

        if 'device' in data and 'sonic_ip' in data['device']:
            list_ucis["sonic_ip.general.enabled"] = data['device']['sonic_ip']['enabled']

            if 'volume' in data['device']['sonic_ip']:
                list_ucis["sonic_ip.general.volume"] = data['device']['sonic_ip']['volume']

        if 'network' in data:
            data_network = data["network"]

            if 'protocol' in data_network:
                web_protocol = data_network['protocol']
                list_ucis[f"httpd.webserver.protocol"] = web_protocol
                proxy_config = data_network["proxy_config"]
                if proxy_config["enable"]:
                    try:
                        list_ucis[f"proxy_config.{web_protocol}.host"]   = data_network["proxy_config"]['host']
                        list_ucis[f"proxy_config.{web_protocol}.port"]  = data_network["proxy_config"]['port']
                        if 'username' in data_network["proxy_config"]:
                            list_ucis[f"proxy_config.{web_protocol}.username"]  = data_network["proxy_config"]['username']
                        if 'noproxy' in data_network["proxy_config"]:
                            list_ucis[f"proxy_config.{web_protocol}.no_proxy"]  = data_network["proxy_config"]['noproxy']

                        if 'password' in data_network["proxy_config"]:
                            plain_password = data_network["proxy_config"]['password']
                            if plain_password:
                                encoded_password = urllib.parse.quote_plus(plain_password)
                                list_ucis[f"proxy_config.{web_protocol}.password"] = encoded_password
                            else:
                                list_ucis[f"proxy_config.{web_protocol}.password"] = ''
                    except:
                        return f"Missing proxy config parameter", 400
            else:
                return 'Missing web protocol', 400

            if data_network['protocol'] == 'http':
                list_ucis[f"proxy_config.http.enable"] = data_network['proxy_config']['enable']
            else:
                list_ucis[f"proxy_config.https.enable"] = data_network['proxy_config']['enable']
                logger.debug(f"data: {data_network}")

            for interface, interface_data in data_network.items():
                if not isinstance(interface_data, dict):
                    continue
                if interface == 'proxy_config':
                    continue
                logger.debug(f"interface: {interface}")

                if 'protocol' not in data_network[interface] and interface != 'wlan0':
                    return f"Missing {interface} protocol", 400
                if 'protocol' in data_network[interface] and interface != 'wlan0':
                    list_ucis[f"network.{interface}.proto"] = data_network[interface]['protocol']
                    if data_network[interface]['protocol'] == 'static':
                        try:
                            list_ucis[f"network.{interface}.ipaddr"]   = data_network[interface]['ip_address']
                            list_ucis[f"network.{interface}.gateway"]  = data_network[interface]['gateway']
                            list_ucis[f"network.{interface}.netmask"]  = data_network[interface]['netmask']
                        except:
                            return f"Missing {interface} parameter", 400
                    elif data_network[interface]['protocol'] == 'dhcp' and 'dhcpname' in data_network[interface]:
                        list_ucis[f"network.{interface}.dhcpname"] = data_network[interface]['dhcpname']
                    logger.info(f"TEST: {data_network[interface]['protocol']} and {data_network[interface]['dhcpname']}")

                if 'dns_type' in data_network[interface]:
                    dns_type_val = data_network[interface]['dns_type']

                    if data_network[interface].get('protocol') == 'static':
                        dns_type_val = 'manual'

                    list_ucis[f"network.{interface}.dns_type"] = dns_type_val

                    if dns_type_val == 'manual':
                        try:
                            dns1 = interface_data['dns1']
                            list_ucis[f"network.{interface}.dns1"] = dns1
                            dns2 = interface_data.get('dns2')
                            if dns2:
                                list_ucis[f"network.{interface}.dns2"] = dns2
                        except:
                            return f"Missing {interface} 'primary dns' parameter", 400

                if 'ssid' in data_network[interface]:
                    ssid = data_network[interface]['ssid']
                    list_ucis[f"network.{interface}.ssid"] = ssid
                    if ssid != '':
                        if 'security' in data_network[interface]:
                            security_type = data_network[interface]['security']
                            list_ucis[f"network.{interface}.security"] = security_type
                            if security_type.upper() != 'NONE':
                                if 'psk' in data_network[interface]:
                                    plain_password = data_network[interface]['psk']

                                    if len(plain_password) < 8:
                                        return f"Password for {interface} must be 8 characters or more", 400

                                    try:
                                        cmd = ['wpa_passphrase', ssid, plain_password]
                                        result = subprocess.run(cmd,
                                                                capture_output=True,
                                                                text=True,
                                                                check=True,
                                                                encoding='utf-8')
                                        match = re.search(r'^\s*psk=([0-9a-fA-F]{64})\s*$', result.stdout, re.MULTILINE)

                                        if match:
                                            hashed_psk = match.group(1)
                                            list_ucis[f"network.{interface}.psk"] = hashed_psk
                                        else:
                                            logger.error(f"Could not parse wpa_passphrase output for {interface}")
                                            return "Error generating WiFi key", 500

                                    except subprocess.CalledProcessError as e:
                                        logger.error(f"wpa_passphrase command failed: {e}")
                                        return "Internal server error (wpa_passphrase missing?)", 500
                                    except Exception as e:
                                        logger.error(f"Error during PSK generation: {e}", exc_info=True)
                                        return "Internal server error", 500
                        else:
                            return f"Missing security parameter for {interface}", 400
                    else:
                        list_ucis[f"network.{interface}.security"] = 'NONE'
                        list_ucis[f"network.{interface}.psk"] = ''

        logger.debug("set ucis")
        for uci in list_ucis:
            logger.debug(uci)

        try:
            logger.debug(f"Total potential UCI changes: {len(list_ucis)}")
            changed_ucis = UciCli.filter_unchanged_ucis(list_ucis)
            logger.debug(f"Actual UCI changes to be applied: {len(changed_ucis)}")

        except UciException as e:
            logger.error(f"UCI error during value filtering: {e}", exc_info=True)
            return "Internal server error (UCI read failed)", 500

        services_to_restart = []

        if not changed_ucis:
            logger.debug("No UCI changes detected. Returning success.")
            return jsonify({'services_restarting': services_to_restart}), 200

        changed_uci_tuples = UciCli.set_uci_configs(changed_ucis)

        logged_ucis = [".".join(t) for t in changed_uci_tuples]

        logger.debug("ucis changed:")
        logger.debug(logged_ucis)

        services_to_restart = Utils.restart_related_services(logged_ucis)

        logger.info("Saved settings values")

        return jsonify({'services_restarting': services_to_restart}), 200

    except Exception as e:
        logger.error(e, exc_info=True)
        return '', 500


@application_bp.route('/system/certificates', methods=['POST'])
@login_required
@update_state_validator
def upload_certificate():
    logger = current_app.logger

    logger.debug(f"request:{request}")
    logger.debug(f"request files:{request.files}")

    if 'uploadfile' not in request.files:
        return jsonify({'message': 'No file part in request.', 'level': 'danger'}), 400

    file = request.files.get('uploadfile')


    if not file or not file.filename:
        return jsonify({'message': 'No file selected.', 'level': 'danger'}), 400

    original_filename = secure_filename(file.filename)
    destination_dir = '/mnt/data/ca-certificates'
    destination_path = os.path.join(destination_dir, original_filename)


    temp_file = None
    try:

        temp_file = tempfile.NamedTemporaryFile(delete=False)
        temp_file_path = temp_file.name
        file.save(temp_file_path)
        temp_file.close()

        logger.info(f"Validating certificate: {original_filename}")
        validation_cmd = ['openssl', 'x509', '-in', temp_file_path]

        validation_result = subprocess.run(validation_cmd,
                                           stdout=subprocess.DEVNULL,
                                           stderr=subprocess.DEVNULL)

        if validation_result.returncode == 0:

            logger.info(f"Validation successful. Installing {original_filename}...")

            os.makedirs(destination_dir, exist_ok=True)
            shutil.move(temp_file_path, destination_path)

            logger.info("Running custom-ca-mgr refresh...")
            refresh_cmd = ['custom-ca-mgr', 'refresh']
            refresh_result = subprocess.run(refresh_cmd,
                                            stdout=subprocess.DEVNULL,
                                            stderr=subprocess.DEVNULL)

            if refresh_result.returncode == 0:
                logger.info(f"Successfully installed {original_filename}")
                return jsonify({'message': f'{original_filename} installed.', 'level': 'good'}), 200
            else:
                logger.error(f"custom-ca-mgr refresh failed for {original_filename}")
                return jsonify({'message': f'{original_filename} installation failed.', 'level': 'danger'}), 500

        else:
            logger.warning(f"Failed to validate {original_filename}. Not a valid certificate.")
            return jsonify({'message': f'{original_filename} is not a valid certificate. Not installed.', 'level': 'danger'}), 400

    except Exception as e:
        logger.error(f'Error processing certificate upload: {e}', exc_info=True)
        return jsonify({'message': 'An internal server error occurred.', 'level': 'danger'}), 500

    finally:
        # 9. Clean up the temp file if it still exists
        #    (i.e., if it wasn't moved)
        if temp_file and os.path.exists(temp_file.name):
            os.remove(temp_file.name)





@application_bp.route('/system/certificates', methods=['GET'])
@login_required
@update_state_validator
def get_certificates_list():
    logger = current_app.logger
    cert_dir = '/mnt/data/ca-certificates'
    result_list = []


    if not os.path.isdir(cert_dir):
        logger.warning(f"Certificate directory not found: {cert_dir}")
        return jsonify({'custom': []})


    try:
        status_cmd = ['custom-ca-mgr', 'status']

        status_result = subprocess.run(
            status_cmd,
            stdout=subprocess.PIPE,
            stderr=subprocess.DEVNULL,
            text=True,
            check=True,
            encoding='utf-8'
        )
        output_lines = status_result.stdout.strip().splitlines()

    except (subprocess.CalledProcessError, FileNotFoundError) as e:
        logger.error(f"Failed to run 'custom-ca-mgr status': {e}", exc_info=True)

        return jsonify({'custom': []})


    for line in output_lines:
        try:
            line = line.strip()

            parts = line.split(' ', 1)

            if len(parts) != 2:
                logger.warning(f"Skipping malformed line from custom-ca-mgr: {line}")
                continue

            state, name = parts
            file_path = os.path.join(cert_dir, name)

            size_str = "Unknown"
            if os.path.exists(file_path):
                try:

                    p1 = subprocess.Popen(
                        ['ls', '-lh', file_path],
                        stdout=subprocess.PIPE,
                        stderr=subprocess.DEVNULL
                    )

                    p2 = subprocess.Popen(
                        ['awk', '{print $5}'],
                        stdin=p1.stdout,
                        stdout=subprocess.PIPE,
                        stderr=subprocess.DEVNULL,
                        text=True,
                        encoding='utf-8'
                    )
                    p1.stdout.close()
                    size_output, _ = p2.communicate()

                    if p2.returncode == 0 and size_output:
                        size_str = size_output.strip()

                except Exception as ls_e:
                    logger.error(f"Could not get size for {name}: {ls_e}", exc_info=True)


            result_list.append({
                'name': name,
                'size': size_str,
                'state': state
            })

        except Exception as e:
            logger.error(f"Error processing certificate line '{line}': {e}", exc_info=True)


    return jsonify({'custom': result_list}), 200


@application_bp.route('/system/certificates/delete', methods=['POST'])
@login_required
@update_state_validator
def delete_certificates():
    logger = current_app.logger

    data = request.get_json()
    if not data or 'filenames' not in data or not isinstance(data['filenames'], list):
        return jsonify({'message': 'Invalid payload. "filenames" list is required.', 'level': 'danger'}), 400

    filenames_to_delete = data.get('filenames', [])
    if not filenames_to_delete:
        return jsonify({'message': 'No filenames provided to delete.', 'level': 'warning'}), 400

    cert_dir = '/mnt/data/ca-certificates'
    abs_cert_dir = os.path.abspath(cert_dir)

    success_count = 0
    failed_files = []

    for filename in filenames_to_delete:
        try:
            if not filename or '..' in filename or filename.startswith('/'):
                logger.warning(f"Skipping potentially malicious filename: {filename}")
                failed_files.append(filename)
                continue

            safe_filename = secure_filename(filename)
            if safe_filename != filename:
                logger.warning(f"Skipping non-secure filename (original: {filename}, sanitized: {safe_filename})")
                failed_files.append(filename)
                continue

            file_path = os.path.join(abs_cert_dir, safe_filename)

            if not os.path.abspath(file_path).startswith(abs_cert_dir):
                logger.warning(f"Skipping path traversal attempt: {filename}")
                failed_files.append(filename)
                continue

            if os.path.exists(file_path):
                os.remove(file_path)
                logger.info(f"Successfully deleted certificate: {filename}")
                success_count += 1
            else:
                logger.warning(f"File not found, skipping delete: {filename}")

        except Exception as e:
            logger.error(f"Failed to delete {filename}: {e}", exc_info=True)
            failed_files.append(filename)

    if success_count > 0:
        logger.info("Refreshing custom CA manager after deletions...")
        try:
            refresh_cmd = ['custom-ca-mgr', 'refresh']
            refresh_result = subprocess.run(
                refresh_cmd,
                stdout=subprocess.DEVNULL,
                stderr=subprocess.PIPE,
                check=True,
                text=True,
                encoding='utf-8'
            )
        except (subprocess.CalledProcessError, FileNotFoundError) as e:
            err_output = e.stderr if hasattr(e, 'stderr') and e.stderr else str(e)
            logger.error(f"custom-ca-mgr refresh failed: {err_output}", exc_info=True)
            return jsonify({
                'message': 'Certificates were deleted, but the CA manager failed to refresh.',
                'level': 'danger'
            }), 500

    if not failed_files:
        return jsonify({
            'message': f'Successfully deleted {success_count} certificate(s).',
            'level': 'good'
        }), 200
    else:
        return jsonify({
            'message': f'Completed with errors. Deleted {success_count} file(s), but failed to delete {len(failed_files)}. Failed files: {", ".join(failed_files)}',
            'level': 'warning'
        }), 207

@application_bp.route('/logs/download', methods=['GET'])
@login_required
@update_state_validator
def web_download_all_logs():
    temp_dir = None
    try:
        try:
            cmd = ['qiba-spi-get-info']
            output = subprocess.check_output(cmd, text=True)
            data = json.loads(output)

            # HW_DEVICE -> Product_Name
            app_name = data.get('HW_DEVICE', {}).get('Product_Name', 'BarixDevice')

        except Exception as e:
            logger.error(f"Failed to get Product_Name from qiba-spi-get-info: {e}")
            app_name = "BarixDevice"
        try:
            with open('/sys/class/net/eth0/address', 'r') as f:
                mac_raw = f.read().strip()
            mac_addr = mac_raw.upper().replace(':', '-')
        except Exception:
            mac_addr = "00:00:00:00:00:00"

        time_tag = datetime.now().strftime("%Y%m%d%H%M%S")

        temp_dir = tempfile.mkdtemp()

        src_log_dir = '/var/log'
        if os.path.exists(src_log_dir):
            for item in os.listdir(src_log_dir):
                s = os.path.join(src_log_dir, item)
                d = os.path.join(temp_dir, item)
                try:
                    if os.path.isdir(s):

                        if item == "diagnostics":
                            continue
                        shutil.copytree(s, d, symlinks=True, ignore_dangling_symlinks=True)
                    else:
                        shutil.copy2(s, d)
                except Exception as e:
                    # Log but continue if we hit a socket or locked file
                    logger.debug(f"Skipped copying {item} from /var/log: {e}")
        # ---------------------------------------------------

        # 3. Create Diagnostics Subdirectory
        diag_path = os.path.join(temp_dir, "diagnostics")
        os.makedirs(diag_path, exist_ok=True)

        # 4. Collect Data (Populate the diagnostics folder)

        # env > environment.txt
        _run_cmd_to_file(['env'], os.path.join(diag_path, 'environment.txt'))

        # uci show > uci_config.txt
        _run_cmd_to_file(['uci', 'show'], os.path.join(diag_path, 'uci_config.txt'))

        # uptime > uptime.txt
        _run_cmd_to_file(['uptime'], os.path.join(diag_path, 'uptime.txt'))

        # top -b -n 5 > top_samples.txt
        _run_cmd_to_file(['top', '-b', '-n', '5'], os.path.join(diag_path, 'top_samples.txt'))

        # df -h > filesystem_usage.txt
        _run_cmd_to_file(['df', '-h'], os.path.join(diag_path, 'filesystem_usage.txt'))

        # cp /proc/meminfo
        if os.path.exists('/proc/meminfo'):
            shutil.copy('/proc/meminfo', os.path.join(diag_path, 'meminfo.txt'))

        # cp /etc/resolv.conf
        if os.path.exists('/etc/resolv.conf'):
            shutil.copy('/etc/resolv.conf', os.path.join(diag_path, 'resolv.conf'))

        # cp /etc/network/interfaces
        if os.path.exists('/etc/network/interfaces'):
            shutil.copy('/etc/network/interfaces', os.path.join(diag_path, 'network_interfaces.txt'))

        # tar czf conf.tar.gz -C /mnt/data conf
        if os.path.exists('/mnt/data/conf'):
            subprocess.run(
                ['tar', 'czf', os.path.join(diag_path, 'conf.tar.gz'), '-C', '/mnt/data', 'conf'],
                check=False
            )
            # Replicate the raw copy: tar cp -R /mnt/data/conf/ $DIAGNOSTICS/conf
            dest_conf = os.path.join(diag_path, 'conf')
            if os.path.exists(dest_conf):
                shutil.rmtree(dest_conf)
            shutil.copytree('/mnt/data/conf', dest_conf)

        # tar czf netstats.tar.gz -C /proc/net .
        if os.path.exists('/proc/net'):
            subprocess.run(
                ['tar', 'czf', os.path.join(diag_path, 'netstats.tar.gz'), '-C', '/proc/net', '.'],
                check=False
            )

        # Factory Info
        with open(os.path.join(diag_path, 'factory_info'), 'w') as f:
            f.write("NO_INFO\n")
        with open(os.path.join(diag_path, 'factory_info.json'), 'w') as f:
            f.write("NO_INFO\n")

        if os.path.exists('/tmp/factory_info_partition/factory_info'):
            shutil.copy('/tmp/factory_info_partition/factory_info', os.path.join(diag_path, 'factory_info'))
        if os.path.exists('/tmp/factory_info_partition/factory_info_json'):
            shutil.copy('/tmp/factory_info_partition/factory_info_json', os.path.join(diag_path, 'factory_info.json'))

        # 5. Create Final Archive
        # Script format: ${DEV_NAME}_${MAC_ADDR}_${TIME_TAG}_logs.tar.gz
        filename = f"{app_name}_{mac_addr}_{time_tag}_logs.tar.gz"
        archive_path = os.path.join(temp_dir, filename) # Temporary location for the archive itself

        # Important: We tar the CONTENTS of temp_dir, which now includes:
        # 1. The original /var/log files
        # 2. The new /diagnostics folder
        # Note: We exclude the archive file itself to avoid infinite recursion if it's in the same dir
        subprocess.run(
            ['tar', 'czf', filename, '--exclude', filename, '.'],
            cwd=temp_dir, # Run tar "inside" the temp dir
            check=True
        )

        final_archive_path = os.path.join(temp_dir, filename)

        # 6. Serve File and Cleanup
        @after_this_request
        def remove_file(response):
            try:
                if temp_dir and os.path.exists(temp_dir):
                    shutil.rmtree(temp_dir)
                    logger.info(f"Cleaned up temp diagnostics directory: {temp_dir}")
            except Exception as e:
                logger.error(f"Error cleaning up temp directory: {e}")
            return response

        logger.info(f"Generated diagnostic logs: {filename}")

        response = make_response(send_file(
            final_archive_path,
            as_attachment=True,
            download_name=filename,
            mimetype='application/gzip'
        ))
        response.headers['Cache-Control'] = 'no-store'
        return response

    except Exception as e:
        logger.error(f"Error generating diagnostic logs: {e}", exc_info=True)
        if temp_dir and os.path.exists(temp_dir):
            shutil.rmtree(temp_dir)
        return jsonify({"error": "Failed to generate logs"}), 500
