import json
import logging
from numbers import Number

import subprocess

from common_lib.lib.JsonRules import JsonRules

logger = logging.getLogger('flask-backend')

def uci_to_json(config_text):
    """
    Simple uci parser
    This doesn't verify if the uci is formed correctly
    """
    config = {}
    section = None

    for line in config_text:
        line = line.strip()
        if not line or line.startswith("#"):  # Skip empty lines and comments
            continue

        if line.startswith("package"):
            continue  # Skip the package declaration

        if line.startswith("config"):
            section = line.split()[-1].strip("'")
            config[section] = {}
        elif section and line.startswith("option"):
            _, key, value = line.split(maxsplit=2)
            key = key.strip()
            value = value.strip("'")
            config[section][key] = value

    # debug
    # print(json.dumps(config, indent=4))

    return config

# TODO - review
def instreamer_set_validate(data):
    mode = data["mode"]

    if mode != "relay":
        codec   = data[mode]["encoder"]["codec"]
        bitrate = data[mode]["encoder"]["bitrate"]
        if data[mode]["encoder"]["bitrate_type"] == "vbr":
            if codec == "aac":
                if not (1 <= bitrate <= 5):
                    return {"message": f"Invalid option for {mode}.encoder.bitrate: Expected value between 1 and 5, got {bitrate}"}, 400
            elif codec == "mp3":
                if not (0 <= bitrate <= 9):
                    return {"message": f"Invalid option for {mode}.encoder.bitrate: Expected value between 0 and 9, got {bitrate}"}, 400
            elif codec == "opus":
                bitrate_list = [8000, 16000, 32000, 48000, 64000, 96000, 128000, 192000, 256000, 320000]
                if not (bitrate in bitrate_list):
                    return {"message": f"Invalid option for {mode}.encoder.bitrate: Expected one of {bitrate_list}, got {bitrate}"}, 400
            elif codec == "vorbis":
                if not (-1 <= bitrate <= 10):
                    return {"message": f"Invalid option for {mode}.encoder.bitrate: Expected value between -1 and 10, got {bitrate}"}, 400

    # Run validation
    with open("application/types/instreamer-ice.rules.json", 'r') as f:
        rules = json.load(f)
    validation_errors = JsonRules.validate_rules(data, rules)
    if validation_errors:
        retval = {
            "message": "Validation errors found",
            "errors": validation_errors
        }
        return retval

    return None




# used to convert uci to json
def instreamer4_uci_to_json(json_data):
    logger.debug(json.dumps(json_data,indent=4))
    instreamer4_input =  {
        "source" :          JsonRules.get_value("instreamer4", "channel_1", "input_source", json_data, "line-in").lower(),

        "usb":{
            "gain" :        0,
            "channel":      JsonRules.get_value("instreamer4", "channel_1", "input_channel", json_data, "mono").lower(),
            "samplerate":   JsonRules.get_value("instreamer4", "channel_1", "sampling_rate", json_data, 44100),
        },
        "line-in":{
            "gain" :        0,
            "channel":      JsonRules.get_value("instreamer4", "channel_1", "input_channel", json_data, "stereo").lower(),
            "samplerate":   JsonRules.get_value("instreamer4", "channel_1", "sampling_rate", json_data, 44100),
        }
    }

    instreamer4_encoder = {
        "codec":        JsonRules.get_value("instreamer4", "enc_1", "type", json_data, "mp3").lower(),
        "bitrate_type": JsonRules.get_value("instreamer4", "enc_1", "bit_rate_type", json_data, "cbr").lower(),
        "bitrate":      JsonRules.get_value("instreamer4", "enc_1", "bit_rate", json_data, 64000)
    }

    instreamer4 = {
        "running":          True,
        "icecast":          {
            "admin_password":   JsonRules.get_value("icecast", "admin", "password", json_data, "barix_icecast"),
        },
        "minijack": {
            "enabled": False,
            "volume": 70,  # value from 0 to 100
        },
        "mode":             "server",
        "server" : {
            "name":         JsonRules.get_value("instreamer4", "icecast_1", "Stream1", json_data, "Barix stream"),
            "input" :       instreamer4_input,
            "encoder" :     instreamer4_encoder,
            "output" : {
                "max_clients":          JsonRules.get_value("instreamer4", "icecast_1", "clients", json_data, 100),
                "mountpoint":           JsonRules.get_value("instreamer4", "icecast_1", "mount_point", json_data, "MyStream1"),
                "username":             JsonRules.get_value("instreamer4", "icecast_1", "user_name", json_data, "source"),
                "password":             JsonRules.get_value("instreamer4", "icecast_1", "password", json_data, "barix_icecast"),
                "icecast_port":         JsonRules.get_value("instreamer4", "icecast_1", "port", json_data, 8000),
                "icecast_name":         JsonRules.get_value("instreamer4", "icecast_1", "icy_name", json_data, "Barix Instreamer ICE"),
                "icecast_description":  JsonRules.get_value("instreamer4", "icecast_1", "icy_description", json_data, "My stream description"),
                "icecast_genre":        JsonRules.get_value("instreamer4", "icecast_1", "icy_genre", json_data, "My genre")
            },
        },
        "source" : {
            "name":         JsonRules.get_value("instreamer4", "icecast_1", "Stream1", json_data, "Barix stream"),
            "input" :       instreamer4_input,
            "encoder" :     instreamer4_encoder,
            "output" : {
                "server":               JsonRules.get_value("instreamer4", "icecast_1", "remote_url", json_data, "http://127.0.0.1"),
                "mountpoint":           JsonRules.get_value("instreamer4", "icecast_1", "mount_point", json_data, "MyStream1"),
                "username":             JsonRules.get_value("instreamer4", "icecast_1", "user_name", json_data, "source"),
                "password":             JsonRules.get_value("instreamer4", "icecast_1", "password", json_data, "barix_icecast"),
                "icecast_port":         JsonRules.get_value("instreamer4", "icecast_1", "port", json_data, 8000),
                "icecast_name":         JsonRules.get_value("instreamer4", "icecast_1", "icy_name", json_data, "Barix Instreamer ICE"),
                "icecast_description":  JsonRules.get_value("instreamer4", "icecast_1", "icy_description", json_data, "My stream description"),
                "icecast_genre":        JsonRules.get_value("instreamer4", "icecast_1", "icy_genre", json_data, "My genre")
            },
        },
        "relay" : {
            "name":                   JsonRules.get_value("instreamer4", "icecast_1", "Stream1", json_data, "Barix stream"),
            "input" : {
                "server":             JsonRules.get_value("icecast", "relay", "server", json_data, ""),
                "port":               JsonRules.get_value("icecast", "relay", "port", json_data, 8000),
                "mountpoint":         JsonRules.get_value("icecast", "relay", "remote_mount", json_data, "input-stream").replace('/',''),
                "username":           JsonRules.get_value("icecast", "relay", "relay_username", json_data, ""),
                "password":           JsonRules.get_value("icecast", "relay", "relay_password", json_data, ""),
            },
            "output" : {
                "max_clients":        JsonRules.get_value("instreamer4", "icecast_1", "clients", json_data, 100),
                "mountpoint":         JsonRules.get_value("icecast", "relay", "local_mount", json_data, "relay-stream").replace('/',''),
                "icecast_port":       JsonRules.get_value("instreamer4", "icecast_1", "port", json_data, 8000),
                "shoutcast_metadata": JsonRules.get_value("icecast", "relay", "relay_shoutcast_metadata", json_data, False),
                "username":           JsonRules.get_value("instreamer4", "icecast_1", "user_name", json_data, "source"),
                "password":           JsonRules.get_value("instreamer4", "icecast_1", "password", json_data, "barix_icecast"),
            },
        }
    }
    return instreamer4
