import functools
import logging
import os

from .exceptions import InvalidFileExtensionError

log = logging.getLogger('flask-backend')

'''
Lists the files that a given directory contains.
@param directoryPath: string containing the full path of the directory from where we want to list the files
Ignores any entry that it's not a file, including files inside nested directories, '.' and '..'.
Returns a list with DirEntry objects corresponding to the files of the given directory or an empty list if the directory doesn't have any file.
'''
def listFilesFromDir(directoryPath):
    fileList = []
    with os.scandir(directoryPath) as it:
        for entry in it:
            if not entry.name.startswith('.') and entry.is_file():
                fileList.append(entry)
    return fileList

'''
Check if a given file has a valid extension, according to the list of extensions allowed provided.
@param filename: string containing the name of the file
@param fileFormatsAllowed: list of strings representing the extensions allowed
If file extension is not valid (it isn't present in the fileFormatsAllowed list), or if it can't get the file extension, raises an InvalidFileExtensionError.
Otherwise, nothing happens.
'''
def checkIfFileExtensionIsAccepted(filename, fileFormatsAllowed):
    try:
        fileExtension = getFileExtension(filename)
        validFormat = False
        for format in fileFormatsAllowed:
            if format.lower() in fileExtension.lower():
                validFormat = True
                break
        if not validFormat:
            raise InvalidFileExtensionError
    except InvalidFileExtensionError:
        raise InvalidFileExtensionError
    except Exception as e:
        log.error(e)
        raise e

'''
Get the extension of a given filename.
@param filename: string representing the file name. Filename should have the format "filename.ext", where ext will be the extension of the file.
If can't split filename by '.', raise InvalidFileExtensionError.
Return a string containing the extension.
'''
def getFileExtension(filename):
    # try:
    #     process = subprocess.Popen(['ffmpeg', '-i', filePath], stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
    #     stdout, stderr = process.communicate()
    #     matches = re.search(r"Audio:\s{1}(?P<format>.+?),", stdout.decode(), re.DOTALL).groupdict()
    #     fileFormat = matches['format']
    # except Exception as e:
    #     log.error(e)
    #     raise InvalidUploadedFileError("invalid file")
    # else:
    #     return fileFormat
    try:
        filenameProperties = filename.split('.')
        if len(filenameProperties) < 2:
            raise InvalidFileExtensionError
    except Exception as e:
        log.error(e)
        raise e
    else:
        return filenameProperties[-1]

'''
Gets the content of a given directory in the form of a tree, including nested directories.
@param rootDir: string containing the fullpath of the directory to analyse.
@param accepted_file_formats: (optional) List of strings containing file extensions accepted.If provided, return only the files that match any of those file extensions.
Returns a list, where each element is a dictionary representing an entry from the directory:
    - If the dict represents a directory, it will have the following keys: name and children, where name is the name of the directory and children will be another list of directories.
    - If the dict represents a valid file, it will have the following keys: name and size, where name is the name of the file and size is the size of the file in bytes.
'''
def getDirectoryContentTree(rootDir, accepted_file_formats=None):
    directoryContentTree = []
    try:
        if os.path.isdir(rootDir):
            with os.scandir(rootDir) as it:
                for entry in it:
                    if not entry.name.startswith('.') and entry.is_file():
                        # check if name has weird characters
                        if not entry.name.isascii():
                            continue
                        try:
                            if accepted_file_formats is not None:
                                checkIfFileExtensionIsAccepted(entry.name, accepted_file_formats)
                            audioFileNode = {}
                            audioFileNode["name"] = entry.name
                            audioFileNode["size"] = os.path.getsize(os.path.join(rootDir, entry.name))
                            directoryContentTree.append(audioFileNode)
                        except InvalidFileExtensionError:
                            pass
                        except Exception as e:
                            log.error(e)
                            raise e
                    if not entry.name.startswith('.') and entry.is_dir():
                        audioFileNode = {}
                        audioFileNode["name"] = entry.name
                        audioFileNode["children"] = getDirectoryContentTree(os.path.join(rootDir, entry.name), accepted_file_formats = accepted_file_formats)
                        directoryContentTree.append(audioFileNode)
        return directoryContentTree
    except Exception as e:
        log.error(e)
        raise e

'''
Sorts a given directory tree list, alphabetically and by directory entry type.
@param treeList: list of dictionaries where each dictionary represents an entry from a directory:
    - If the dict represents a directory, it will have the following keys: name and children, where name is the name of the directory and children will be another list of directories.
    - If the dict represents a valid file, it will have the following keys: name and size, where name is the name of the file and size is the size of the file in bytes.
@param direction: string indicating the direction of the sorting function:
    - If direction is ASC, files will show up before directories.
    - If direction is not ASC, files will show up after directories.
    - In case both items are from the same type (both directories or both files), they'll be sorted alphabetically. 
Returns a new list with the provided treeList sorted.
'''
def sortDirectoryTreeList(treeList, direction="ASC"):
    if direction == "ASC":
        for elem in treeList:
            if 'children' in elem.keys():
                elem['children'] = sortDirectoryTreeList(elem['children'])
        sortedTreeList = sorted(treeList, key=functools.cmp_to_key(sortDirectoryTree))
    else:
        for elem in treeList:
            if 'children' in elem.keys():
                elem['children'] = sortDirectoryTreeList(elem['children'], direction='DESC')
        sortedTreeList = sorted(treeList, key=functools.cmp_to_key(sortDirectoryTree), reverse=True)
    return sortedTreeList


'''
Custom sorting function to sort a dictionary representing a directory tree.
'''
def sortDirectoryTree(item1, item2):
    if 'children' in item1.keys() and 'children' in item2.keys(): # both directories
        if item1["name"].lower() < item2["name"].lower():
            return -1
        else:
            return 1
    elif 'children' in item1.keys() and 'children' not in item2.keys(): # item1 is directory and item2 is
        return 1
    elif 'children' not in item1.keys() and 'children' in item2.keys():
        return -1
    else:
        if item1["name"].lower() < item2["name"].lower():
            return -1
        else:
            return 1