from enum import Enum, IntEnum
import json
from pathlib import Path
from typing import Union


class BarixReturn(Enum):
    SUCCESS = 0
    FAILED = 1
    NOT_IMPLEMENTED = 2
    ENCODING_ERROR = 3
    DECODING_ERROR = 4
    BRTP_EXTENSION_SET = 5
    BRTP_EXTENSION_RESET = 6
    RUNNING = 7
    NOT_RUNNING = 8
    SAME_VALUE = 9
    NOT_INITIALIZED = 10
    NETWORK_IS_UNREACHABLE = 101

    def __str__(self):
        return f'{self.name}'


class BarixHWType(IntEnum):
    """
        As defined here:
        https://barix.atlassian.net/wiki/spaces/K1002/pages/4477849598/IPAM+Types+HW+Types+and+SNMP+MIBs
    """
    ANI_CONSUMER_MAINBOARD = 1  # Legacy 1
    ANI_CONSUMER_MAINBOARD_MOH = 2  # Legacy 2
    ANI_COMMERCIAL_MAINBOARD_FULL = 3  # Legacy 3
    ANI_COMMERCIAL_MAINBOARD_PCMCIA = 4  # Legacy 4
    ANI_COMMERCIAL_MAINBOARD_STD = 5  # Legacy 5
    ANI_NET7000_MAINBOARD = 6  # Legacy 6
    ANI_ANNUNCICOMIC_MAINBOARD = 7  # Annuncicom 100
    ANI_INSTREAMER_MAINBOARD = 8  # Instreamer 100
    ANI_EXSTREAMERDIG_MAINBOARD = 9  # Exstreamer Digital
    ANI_FLCOMSERVER_MAINBOARD = 10
    ANI_ACEEX_MAINBOARD = 11
    ANI_QUINX_SCDS_MAINBOARD = 12
    ANI_IP_AUDIO_MODULE = 13  # IPAM 100
    ANI_IP_EXSTREAMER_MODULE = 14  # Exstreamer 100
    ANI_EXSTREAMER_200 = 15  # Exstreamer 200
    IPAM_EVAL_KIT = 16  # IPAM Evaluation kit
    ANI_ANNUNCICOM_1000 = 17  # Annuncicom 1000
    BOSCH = 18  # Bosch IP Audio
    ANI_ANNUNCICOM_200 = 19  # Annuncicom 200
    ANI_EXSTREAMER_110 = 20  # Exstreamer 110
    ANI_EXSTREAMER_1000 = 21  # Exstreamer 1000
    BARIONET_200 = 22  # Barionet 200
    DIGIGRAM_PYKO_OUT = 23
    DIGIGRAM_PYKO_IN = 24
    NEUMANN_DS6 = 25  # Annuncicom PS16
    BARIONET_50 = 26  # Barionet 50
    ANNUNCICOM_200_AIPHONE = 27
    FLCOMSERVER_BASIC_FLCOMSERVER_GEN2 = 28
    FLCOMSERVER_UNI = 29
    FLCOMSERVER_PRO = 30
    FLCOMSERVER_PSI_MODEM = 31
    ANNUNCICOM_155 = 32  # Annuncicom 155
    ANNUNCICOM_VME = 33  # Annuncicom VME
    EXSTREAMER_120 = 34  # Exstreamer 120
    EXSTREAMER_500 = 35  # Exstreamer 500
    IP_SPEAKER = 36  # Exstreamer P5
    EXSTREAMER_105 = 37  # Exstreamer 105
    EXSTREAMER_205 = 38  # Exstreamer 205
    IPAM_101 = 39  # IPAM 101
    IPAM_102 = 40  # IPAM 102
    ANNUNCICOM_50 = 41  # Annuncicom 60
    ANNUNCICOM_PS1 = 42  # Annuncicom PS1
    INSTREAMER_V3 = 43  # Instreamer
    ANNUNCICOM_50_OEM = 44  # Annuncicom 50 OEM
    IPAM_301 = 45  # IPAM 301
    IPAM_302 = 46  # IPAM 302
    BARIONET_50_AIPHONE = 47  # Barionet 50 OEM
    EXSTREAMER_100L = 48  # Exstreamer 100L
    AP_ENCODER = 49  # Audio Point ENC
    AP_CONTROLLER = 50  # Audio Point CTL
    OEM_362 = 51  # OEM 362
    LINULUS = 52  # Linulus
    Barix_IPW_2A = 53  # IPW-2A / IX-1AS v2
    Barix_M400 = 54  # Barix M400
    Barix_L400 = 55  # Barix L400
    Barix_MA400 = 56  # Barix MA400
    Barix_MX400 = 57  # Barix MX400
    QINO_398_BOARD = 58  # QINO 398 Board
    Barix_S400 = 59  # Barix S400
    Barix_M401 = 60  # Barix M401
    Terma_VEP = 61  # TERMA Voice End Point
    Barionet_1100 = 62  # Barionet 1100
    Barionet_200 = 63  # Barionet 200
    Barionet_400 = 64  # Barionet 400
    Barix_SP400 = 65  # Barix SP400
    Barix_TPA400 = 66  # Barix TPA400
    Redbox_Intercom = 67  # Redbox Intercom System
    USSI_Encompass_Controller = 68  # USSI Encompass Controller
    Barix_MPA400 = 69  # Barix MPA400
    Barix_L408 = 70  # Multin L408
    EdgePlayer = 71  # iHeart EdgePlayer
    Barix_PS_Touch_XTR = 72  # Barix Annuncicom PS Touch XTR
    Barix_MPI400 = 73  # Barix MPI400
    Barionet_1000 = 74  # Barionet 1000
    Hyperspike_MPA400 = 75  # Hyperspike MPA400
    LOX400 = 76  # Synstreamer LOX400
    Barionet_M41 = 77  # Barionet M41
    Bavidi_Device = 78  # Bavidi
    MS700 = 79  # AE MS700
    LX400 = 80  # Barix LX400
    Unigate = 81  # Qiba Unigate
    BARIONET_100 = 93  # Barionet
    DIGIGRAM_TEMP = 99

    def __str__(self):
        return f'{self.name}'


class BarixIPAMType(IntEnum):
    UNKNOWN_GENERIC = 0
    IPAM_101 = 1
    IPAM_102 = 2
    IPAM_301 = 3
    IPAM_302 = 4
    IPAM_390 = 5
    IPAM_300 = 6  # rev >= 1.4
    QINO = 7
    IPAM_400 = 8

    def __str__(self):
        return f'{self.name}'


class BarixExitCode(Enum):
    EX_OK = 0               # successful termination
    EX_USAGE = 64           # command line usage error
    EX_DATAERR = 65         # data format error
    EX_NOINPUT = 66         # cannot open input
    EX_NOUSER = 67          # addressee unknown
    EX_NOHOST = 68          # host name unknown
    EX_UNAVAILABLE = 69     # service unavailable
    EX_SOFTWARE = 70        # internal software error
    EX_OSERR = 71           # system error (e.g., can't fork)
    EX_OSFILE = 72          # critical OS file missing
    EX_CANTCREAT = 73       # can't create (user) output file
    EX_IOERR = 74           # input/output error
    EX_TEMPFAIL = 75        # temp failure; user is invited to retry
    EX_PROTOCOL = 76        # remote error in protocol
    EX_NOPERM = 77          # permission denied
    EX_CONFIG = 78          # configuration error

    def __str__(self):
        return f'{self.name}'


class BarixRTPPayloadTypes(IntEnum):
    MULAW_8kHz_MONO = 0
    ALAW_8kHz_MONO = 8
    G722_16kHz_MONO = 9
    PCM_MSB_44kHz_STEREO = 10
    PCM_MSB_44kHz_MONO = 11
    MP3_44kHz_STEREO = 14
    PCM_MSB_8kHz_MONO = 96
    MULAW_24kHz_MONO = 97
    ALAW_24kHz_MONO = 98
    PCM_MSB_24kHz_MONO = 99
    MULAW_32kHz_MONO = 100
    ALAW_32kHz_MONO = 101
    PCM_MSB_32kHz_MONO = 102
    PCM_MSB_32kHz_STEREO = 103
    PCM_LSB_8kHz_MONO = 104
    PCM_LSB_24kHz_MONO = 105
    PCM_LSB_32kHz_MONO = 106
    PCM_LSB_44kHz_STEREO = 107
    PCM_LSB_48kHz_STEREO = 108
    MULAW_12kHz_MONO = 109
    ALAW_12kHz_MONO = 110
    PCM_MSB_12kHz_MONO = 111
    PCM_LSB_12kHz_MONO = 112
    METADATA = 114
    AAC = 124
    OPUS = 125
    AAC_PLUS = 126
    GENERIC = 127  # Use to support other types. SDP must be used in this case

    def __str__(self):
        return f'{self.value}'


class ReflectorDeviceModes(Enum):
    MODE_NONE = 0
    MODE_ENCODER = 1
    MODE_DECODER = 2

    def __str__(self):
        return f'{self.name}'


class ReflectorInputSource(Enum):
    INPUT_LINE = 1
    INPUT_MIC = 2
    INPUT_OPTICAL = 4
    INPUT_COAX = 5

    def __str__(self):
        return f'{self.name}'


def export_json(write_to_path: Union[None, str, Path] = None) -> None:
    """
        Generate a JSON file with all the enums contained in the barix_enums class.
        If no parameter is given, the json file is created in the same directory as the class itself
    @param write_to_path: Filesystem path, where the json file to be stored.
                        If it is given as a string, it must be the full path with the filename
    @return:
    """

    if isinstance(write_to_path, str):
        write_to_path = Path(write_to_path)
        print(f'Using given filepath: {write_to_path.absolute()}')
        if not write_to_path.exists():
            print(f"Provided path: {write_to_path.absolute()} does not exist. Using default")
            write_to_path = Path('./barix_enums.json')
    elif write_to_path is None:
        write_to_path = Path('./barix_enums.json')
        print(f"Using default path: {write_to_path.absolute()}")

    print("Generating JSON file with all available Barix enums...")
    barix_enums = dict.fromkeys(
        ['hw_types', 'rtp_payload_types', 'reflector_input_source', 'reflector_device_modes', 'ipam_types']
    )
    hw_types_list = list(range(len(BarixHWType)))
    for idx, entry in enumerate(BarixHWType):
        hw_types_list[idx] = {'id': entry.value, 'name': entry.name}

    rtp_payload_types_list = list(range(len(BarixRTPPayloadTypes)))
    for idx, entry in enumerate(BarixRTPPayloadTypes):
        rtp_payload_types_list[idx] = {'id': entry.value, 'name': entry.name}

    reflector_input_source_list = list(range(len(ReflectorInputSource)))
    for idx, entry in enumerate(ReflectorInputSource):
        reflector_input_source_list[idx] = {'id': entry.value, 'name': entry.name}

    reflector_device_modes_list = list(range(len(ReflectorDeviceModes)))
    for idx, entry in enumerate(ReflectorDeviceModes):
        reflector_device_modes_list[idx] = {'id': entry.value, 'name': entry.name}

    ipam_types_list = list(range(len(BarixIPAMType)))
    for idx, entry in enumerate(BarixIPAMType):
        ipam_types_list[idx] = {'id': entry.value, 'name': entry.name}

    barix_enums['hw_types'] = hw_types_list
    barix_enums['rtp_payload_types'] = rtp_payload_types_list
    barix_enums['reflector_input_source'] = reflector_input_source_list
    barix_enums['reflector_device_modes'] = reflector_device_modes_list
    barix_enums['ipam_types'] = ipam_types_list

    with open(write_to_path, 'w') as f:
        f.write(json.dumps(barix_enums, indent=4))
        if isinstance(write_to_path, Path):
            print(f"File can be found at: {write_to_path.absolute()}")
        else:
            print(f"File can be found at: {write_to_path}")

    print("Done!")


# if __name__ == "__main__":
#     """
#         Use this to test the generation of the json file
#     """
#     file_path = './barix_enums.json'
#     export_json(file_path)
