import os, logging

logger = logging.getLogger(__name__)

class Gpio:
    '''
    Module to handle with GPIOs. It allows read and write operations.
    This module works on top of GPIOs sysfs from the Linux kernel
    (https://www.kernel.org/doc/Documentation/gpio/sysfs.txt).
    '''
    GPIO_ROOT = '/sys/class/gpio'
    GPIO_EXPORT = os.path.join(GPIO_ROOT, 'export')
    gpiopath = lambda self, pin: os.path.join(self.GPIO_ROOT, 'gpio{0}'.format(pin))

    IN, OUT = 'in', 'out'
    DIRECTION = ['in', 'out', 'low', 'high']

    def __init__(self, pin: str, mode: str, initial:bool=False, active_low:bool=False):
        '''
        Instantiate a GPIO object.

        :param str pin: The GPIO indentifier. Normally, a string in the form of 'gpioN', e.g., gpio19.
        :param str mode: 'in' to configure the GPIO as an input. 'out' to configure the GPIO as an output.
        :param bool initial: Initial value for the GPIO if configured as an output.
        :param bool active_low: False, to invert the value attribute for both reading and writing operations.

        Raises:
            TypeError: if 'pin' is not a string
            ValueError: if 'mode' value is not supported
        '''
        if not isinstance(pin, str):
            raise TypeError(pin)
        self._pin = pin

        if mode not in (self.IN, self.OUT):
            raise ValueError(mode)

        self._ppath = self.gpiopath(pin)
        if not os.path.exists(self._ppath):
            logger.debug("Creating GPIO {0}".format(pin))
            with open(self.GPIO_EXPORT, "w") as f:
                f.write(pin)

        self._value     = None
        self._direction = None

        if mode == self.OUT:
            self.setDirection("high" if initial else "low")
        else:
            self.setDirection(mode)

        with open(os.path.join(self._ppath, "active_low"), "w+") as f:
            if active_low:
                f.write('1')
            else:
                f.write('0')


    @property
    def fileno(self) -> int:
        '''
        Return the 'value' file descriptor of the underlying kernel sysfs.

        :rtype: int
        :return: The file descriptor integer.
        '''
        return self._value.fileno()

    def read(self) -> int:
        '''
        Perform a read operation of the GPIO value.

        :rtype: int
        :return: The current value for the GPIO.
        '''
        self._value.seek(0)
        val = int(self._value.read().strip())
        #logger.debug('Read {0}: {1}'.format(self._pin, val))
        return val


    def write(self, value:int):
        '''
        Perform a write operation on GPIO value. Only for 'in' GPIOs.

        :param int value: Value to write in the GPIO.
        '''
        if self._direction not in ("out", "high", "low"):
            return
        value = int(bool(value))
        #logger.debug('Write {0}: {1}'.format(self._pin, value))
        self._value.write(str(value))
        self._value.flush()


    def setDirection(self, direction: str) -> None:
        """
        Sets direction:
          * in, out, low, high
        """
        if direction not in self.DIRECTION:
            raise ValueError(mode)

        if direction==self._direction:
            return

        self._direction = direction

        if self._value is not None:
            self._value.close()

        with open(os.path.join(self._ppath, "direction"), "w+") as f:
            f.write(self._direction)

        if self._direction == self.IN:
            fileMode = "r"
        else:
            fileMode = "w+"

        try:
            self._value = open(os.path.join(self._ppath, "value"), fileMode)
        except Exception as e:
            if self._value:
                self._value.close()
                raise e

