
#  TAS6421 Digital Amplifier
#
#

import time
import subprocess
import json
from threading import Thread
from . audio_amp import *

class TAS6421(AudioAmp):

    def __init__(self, i2c_bus, chip_address, amp_vol):
        super(TAS6421, self).__init__(i2c_bus,chip_address)
        self.amp_vol = amp_vol
        self.gain = 0x30
        self.speaker_detect_imp = 0
        self.otw = 0

    def resetAmp(self):
        self.writeByte(0x00, 0x80)
        print("Reset TAS6421 amp")

    def initAmp(self):
        self.writeByte(0x00, 0x00)  # default on reset
        self.writeByte(0x01, self.gain) 
        self.writeByte(0x02, 0x66)  # set 128x FSCLK
        self.writeByte(0x03, 0x04)  # default on reset
        self.writeByte(0x05, 0x00)  #mute the PCM Volume at start
        self.writeByte(0x09, 0x00)  # default on reset
        self.writeByte(0x0a, 0x11)  # default on reset
        # self.writeByte(0x0c, 0x00)  # default on reset
        # self.writeByte(0x0e, 0x00)  # default on reset
        # 0x0f is state register, to read only. At reset must be 0x40
        # self.writeByte(0x0f, 0x40)
        self.writeByte(0x14, 0x00)  # default on reset
        self.writeByte(0x15, 0x00)  # default on reset
        self.writeByte(0x16, 0x00)  # default on reset
        # self.writeByte(0x17, 0x00)  # default on reset
        # self.writeByte(0x1b, 0x00)  # default on reset
        # self.writeByte(0x1c, 0x00)  # default on reset
        self.writeByte(0x21, 0x00)  # default on reset
        self.writeByte(0x22, 0x01)  # default on reset
        self.writeByte(0x23, 0x14)  # default on reset
        self.writeByte(0x24, 0x00)  # default on reset
        #setting pcm volume
        print("Set pcm volume to: {}".format(hex(self.amp_vol)))
        self.writeByte(0x05, self.amp_vol)
        print("TAS6421 initialized!")

    def powerOnAmp(self):
        self.writeByte(0x04, 0x15)
        print("TAS6421 amplifier powered on!")

    def setupAmp(self):
        self.resetAmp()
        time.sleep(1)
        self.initAmp()
        time.sleep(0.002)
        self.powerOnAmp()

    def setVol(self, vol=None):
        if vol is not None:
            self.amp_vol = vol
        self.writeByte(0x05, self.amp_vol)
        print("Set Amp volume to: {}".format(hex(self.amp_vol)))

    def getVol(self):
        return hex(self.readByte(0x05))
    
    def setGain(self, gain=None):
        if gain is not None:
            self.gain = gain
        self.writeByte(0x01, self.gain)
        print("Set Amp gain to: {}".format(hex(self.gain)))

    def getGain(self):
        return hex(self.readByte(0x1))


    """
    Impedance Magnitude Measurement (check procedure in section 9.3.7.3.1 of TAS6421-Q1 datasheet)

    """
    def impedanceMeasurement(self):
        # Set the output channel into the Hi-Z state
        self.writeByte(0x04, 0x55)
        # Set the AC_DIAGS_LOOPBACK bit (bit 7 in register 0x16) to 0
        # use I= 19mA
        self.writeByte(0x16, 0x40)
        # Apply a full-scale input signal from the DSP for the tested channel
        # with the desired frequency (recommended 10 khz to 20 khz)
        proc = subprocess.Popen("/usr/bin/aplay -Dplug:digital_out /usr/share/sounds/19kH_stereo_5sec.wav &",shell=True)
        time.sleep(.5)
        #  Set the device into the AC diagnostic mode (set bit 3 in register 0x15 to 1)
        self.writeByte(0x15, 0x08)
        #wait Channel reporting register indicates the status changing:
        # From "AC Diagnostic Mode" TO "Hi-Z" state
        counter=0
        while self.readByte(0x0F) != 0x40:
            counter+=1
            print("ReadCounter:{}".format(counter))
            if counter == 50:
                print("Error: cannot read/write TAS6421 Amp")
                break	
            time.sleep(0.5)

        ch_imp=self.readByte(0x17)
        proc.terminate()
        print("Channel Impedance: {}".format(hex(ch_imp)))
        self.speaker_detect_imp = "%0.2f" % ((ch_imp*2.371)/19)
        print("Speaker Impedance:{}".format(self.speaker_detect_imp))

    def getImpedance(self):
        return self.speaker_detect_imp

    """
    select TAS-6421 gain and volume based on the power source and speaker impedance
    the json file has all the possible scenarios 
    """
    def selectGainVol(self,power_source, speaker_imp, json_file):
        with open(json_file, 'r') as f:
            content = json.load(f)

        print(content)
        if speaker_imp == "auto":
            print("Use detected speaker impedance")
            detect_imp = float(self.speaker_detect_imp)
            speaker_imp = 4
            if detect_imp > 3.5 and detect_imp < 5 :
                speaker_imp = 4
            elif detect_imp > 7 and detect_imp <= 10 :
                speaker_imp = 8

        for data in content['amp_settings']:
            if power_source == data['power_source'] and int(speaker_imp) == int(data['speaker_imp']):
                self.gain = int(data['gain'],16)
                self.amp_vol = int(data['vol_ctrl'],16)
                print("selected gain:{}, vol:{}".format(hex(self.gain),hex(self.amp_vol)))
                break
        
    def getOverTemp(self):
        warnRegister = self.readByte(0x13)
        print("Warning Register:{}".format(hex(warnRegister)))
        otw = (warnRegister & 0x10) >> 4
        return otw
