#!/usr/bin/python3

#==================================================================================
#   Dante License Check 
#   
#   Used to check and handle Dante License
#   Handle leds during the Dante activation phase
#   (Depending on device used the led behavior may need to be addapted accordingly)
#==================================================================================

import json
import shutil
import sys
import os
import logging
import re
import socket
import subprocess
import time
from enum import Enum
from threading import Thread
from barix.io import leds
from barix.system.barix_enums import BarixHWType

logger = logging.getLogger(__name__)

class DeviceActivationState(Enum):
    IDLE = 0
    INIT = 1
    ERROR = 2
    FINISH = 3

class DanteCheck:

    DANTE_ACTIVATION_FOLDER = "/mnt/data/dep/dante_package/dante_data/activation"
    DANTE_LICENSE           = "device.lic"
    DANTE_MANUFACTURER      = "manufacturer.cert"

    def __init__(self):
        self.led1 = None
        self.led2 = None
        self.server_socket = None
        self.stop_running = False
        self.state = DeviceActivationState.IDLE
        self.led_thread = None
        hwType = self.getHwType()
        self.using2Leds = True
        if hwType in [BarixHWType.Barix_S400.value, BarixHWType.Barix_SP400.value]: #one led device
            self.using2Leds = False

    def start(self):
        if not self.isDanteFeatureAvailable():
            logger.info("Barix's Dante feature not found ... exiting application")
            sys.exit(0)

        if os.path.isfile(f"/mnt/shadow/{self.DANTE_LICENSE}"):
            if not self.checkDanteLicense():
                self.updateDepLicenseFromShadow()
                proc = subprocess.Popen("reboot", shell=True)
                proc.wait()
            logger.info("License already on /mnt/shadow ... exiting application")
            sys.exit(0)

        # leds init
        self.led1 = leds.IpamLed("led1")
        if self.using2Leds:
            self.led2 = leds.IpamLed("led2")

        self.led_thread = Thread(target=self.ledOperation)
        self.led_thread.start()

        self.initTcpServer()

        self.runLoop()

    def isDanteFeatureAvailable(self):
        jsonFile = open('/tmp/factory_info_partition/factory_info_json', 'r')
        config_json = json.load(jsonFile)
        for result in config_json['SW_LICENSES']:
            if re.search('Dante', result['Image_Features']):
                logger.info("Found Barix's Dante feature")
                return True
        return False

    def updateShadowLicense(self):
        try:
            p = subprocess.Popen("mount -o remount,rw /mnt/shadow",shell=True)
            p.wait()
            shutil.copy(f"{self.DANTE_ACTIVATION_FOLDER}/{self.DANTE_LICENSE}","/mnt/shadow")
            shutil.copy(f"{self.DANTE_ACTIVATION_FOLDER}/{self.DANTE_MANUFACTURER}", "/mnt/shadow")
            p = subprocess.Popen("sync",shell= True)
            p. wait()
            p = subprocess.Popen("mount -o remount,ro /mnt/shadow", shell=True)
            p.wait()
        except Exception as e:
            logger.error(f"Update Shadow License:{e}")

    def checkDanteLicense(self):
        if os.path.isfile(f"{self.DANTE_ACTIVATION_FOLDER}/{self.DANTE_LICENSE}"):
            logger.info("Dante license is already DEP activation folder")
            return True
        logger.info("Dante license NOT found in the DEP activation folder")
        return False

    def updateDepLicenseFromShadow(self):
        try:
            shutil.copy(f"/mnt/shadow/{self.DANTE_LICENSE}", f"{self.DANTE_ACTIVATION_FOLDER}")
            shutil.copy(f"/mnt/shadow/{self.DANTE_MANUFACTURER}", f"{self.DANTE_ACTIVATION_FOLDER}")
            p = subprocess.Popen("sync",shell= True)
            p. wait()
        except Exception as e:
            logger.error(f"Update Dep License:{e}")

    def initTcpServer(self):
        # Init TCP server
        host = '0.0.0.0'
        port = 6666
        logger.info(f"Start TCP socket")
        self.server_socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.server_socket.bind((host, port))
        self.server_socket.listen(1)
        logger.info(f"Listening for incoming connections on {host}:{port}")

    def ledOperation(self):
        while True:
            if self.state == DeviceActivationState.INIT:
                self.ledInitState()

            if self.state == DeviceActivationState.ERROR:
                self.ledResultState(False)

            if self.state == DeviceActivationState.FINISH:
                self.ledResultState(True)

            if self.state == DeviceActivationState.IDLE:
                time.sleep(1)

    def ledInitState(self):
        self.led1.set("yellow", blink=False)
        if self.using2Leds:
            self.led2.set("off", blink=False)
        time.sleep(1)
        self.led1.set("off", blink=False)
        if self.using2Leds:
            self.led2.set("yellow", blink=False)
        time.sleep(1)

    def ledResultState(self, success:bool):
        if success:
            self.led1.set("green", blink=False)
            if self.using2Leds:
                self.led2.set("green", blink=False)
            time.sleep(0.05)
            self.led1.set("off", blink=False)
            if self.using2Leds:
                self.led2.set("off", blink=False)
            time.sleep(0.05)
        else:
            self.led1.set("red", blink=False)
            if self.using2Leds:
                self.led2.set("red", blink=False)
            time.sleep(0.05)
            self.led1.set("off", blink=False)
            if self.using2Leds:
                self.led2.set("off", blink=False)
            time.sleep(0.05)

    def getHwType(self):
        p = subprocess.Popen(["/usr/bin/qiba-spi-get-production-info.sh", "-w"], stdout=subprocess.PIPE, shell=False)
        return int(p.communicate()[0]) 

    def runLoop(self):
        while not self.stop_running:
            client_socket, client_address = self.server_socket.accept()
            logger.info(f"Connection from {client_address}")

            stop_reading = False
            while not stop_reading:
                data = client_socket.recv(1024).decode('utf-8')
                logger.info(f"Read:{data}")

                if len(data) <= 0:
                    stop_reading = True
                    logger.info("Stop Reading...")

                if data == 'INIT':
                    logger.info("Init Dante License Activation checking -- Blink Orange leds")
                    self.state = DeviceActivationState.INIT
                    continue

                if data == 'ERROR':
                    logger.error("Failed Dante License Activation procedure -- Blink red leds")
                    self.state = DeviceActivationState.ERROR
                    continue

                if data == 'FINISH':
                    self.updateShadowLicense()
                    logger.info("Copy license files to /mnt/shadow")
                    proc = subprocess.Popen("sync", shell=True)
                    proc.wait()
                    logger.info("Finish Dante License Activation with success -- Blink green leds")
                    self.state = DeviceActivationState.FINISH
                    continue

            client_socket.close()

        logger.info("Finish application with success!")
        sys.exit(0)


if __name__ == "__main__":
    logging.basicConfig(filename='/var/log/dante-license-check.log', level=logging.INFO, format='%(asctime)s %(name)s %(levelname)s %(message)s')
    logger.info("Start dante_check application")

    app = DanteCheck()
    app.start()
