#!/usr/bin/python3
# 
# Set DEP's dante.json configuration file from UCI configuration
#

from uci import Uci
import json
import logging

logger = logging.getLogger(__name__)

class DanteCfgGenerator:

    DEP_DANTE_CFG = "/mnt/data/dep/dante_package/dante_data/capability/dante.json"

    def __init__(self) -> None:
        self.depEnabled = None
        self.txChannels = None
        self.rxChannels = None
        self.sampleRate = None
        self.availableSampleRates = None
        self.supportedEncodings = None
        self.defaultEncoding = None
        self.networkInterfaces = None
        self.networkLinkSpeed = None
        self.jsonConfig = None

    def start(self):
        if self.__checkUciConfig():
            self.__readCfgFile()
            if self.jsonConfig is None:
                logger.error("Cannot open dante.json file")
            self.__compareConfig()  


    def __uciGet(self, uci: Uci, config: str, section: str, opt: str) -> str:
        try:
            return uci.get(config,section,opt)
        except:
            return None

    def __checkUciConfig(self) -> bool:
        uci = Uci()
        self.depEnabled = self.__uciGet(uci,"dante","dep","enabled")
        if self.depEnabled is None:
            logger.error("Missing dante.dep.enabled UCI config")
            return False
        
        if self.depEnabled == "false":
            logger.info("DEP is disabled")
            return False

        self.txChannels = self.__uciGet(uci,"dante","dep","tx_channels")
        if self.txChannels is None:
            logger.error("Missing dante.dep.tx_channels UCI config")
            return False

        self.rxChannels = self.__uciGet(uci,"dante","dep","rx_channels")
        if self.rxChannels is None:
            logger.error("Missing dante.dep.rx_channels UCI config")
            return False

        self.sampleRate = self.__uciGet(uci,"dante","dep","sample_rate")
        if self.sampleRate is None:
            logger.error("Missing dante.dep.sample_rate UCI config")
            return False

        sampleRates = self.__uciGet(uci,"dante","dep","available_sample_rates")
        if sampleRates is None:
            logger.error("Missing dante.dep.available_sample_rates UCI config")
            return False
        else:
            self.availableSampleRates = [int(x) for x in sampleRates.split(',')]
            
        encodings = self.__uciGet(uci,"dante","dep","supported_encodings")
        if encodings is None:
            logger.error("Missing dante.dep.supported_encodings UCI config")
            return False
        else:
            self.supportedEncodings = encodings.split(',')

        self.defaultEncoding = self.__uciGet(uci,"dante","dep","default_encoding")
        if self.defaultEncoding is None:
            logger.error("Missing dante.dep.default_encoding UCI config")
            return False
        
        interfaces = self.__uciGet(uci,"dante","dep","network_interfaces")
        if interfaces is None:
            logger.error("Missing dante.dep.network_interfaces UCI config")
            return False
        else:
            self.networkInterfaces = interfaces.split(',')

        self.networkLinkSpeed = self.__uciGet(uci,"dante","dep","preferred_link_speed")
        if self.networkLinkSpeed is None:
            logger.error("Missing dante.dep.preferred")

        return True
    
    def __readCfgFile(self):
        with open(self.DEP_DANTE_CFG) as jsonFile:
            self.jsonConfig = json.load(jsonFile)

    def __compareConfig(self):
        changed = False
        try:
            if self.jsonConfig is not None:
                if self.jsonConfig['audio']['txChannels'] != int(self.txChannels):
                    logger.warning(f"txChannels changed! new: {self.txChannels} old: {self.jsonConfig['audio']['txChannels']}")
                    self.jsonConfig['audio']['txChannels'] = int(self.txChannels)
                    changed = True

                if self.jsonConfig['audio']['rxChannels'] != int(self.rxChannels):
                    logger.warning(f"rxChannels changed! new: {self.rxChannels} old: {self.jsonConfig['audio']['rxChannels']}")
                    self.jsonConfig['audio']['rxChannels'] = int(self.rxChannels)
                    changed = True

                if self.jsonConfig['audio']['sampleRate'] != int(self.sampleRate):
                    logger.warning(f"sampleRate changed! new: {self.sampleRate} old: {self.jsonConfig['audio']['sampleRate']}")
                    self.jsonConfig['audio']['sampleRate'] = int(self.sampleRate)
                    changed = True

                if self.jsonConfig['audio']['availableSampleRates'] != self.availableSampleRates:
                    logger.warning(f"availableSampleRates changed! new: {self.availableSampleRates} old: {self.jsonConfig['audio']['availableSampleRates']}")
                    self.jsonConfig['audio']['availableSampleRates'] = self.availableSampleRates
                    changed = True

                if self.jsonConfig['audio']['supportedEncodings'] != self.supportedEncodings:
                    logger.warning(f"supportedEncodings changed! new: {self.supportedEncodings} old: {self.jsonConfig['audio']['supportedEncodings']}")
                    self.jsonConfig['audio']['supportedEncodings'] = self.supportedEncodings
                    changed = True

                if self.jsonConfig['audio']['defaultEncoding'] != self.defaultEncoding:
                    logger.warning(f"defaultEncoding changed! new: {self.defaultEncoding} old: {self.jsonConfig['audio']['defaultEncoding']}")
                    self.jsonConfig['audio']['defaultEncoding'] = self.defaultEncoding
                    changed = True

                if self.jsonConfig['network']['interfaces'] != self.networkInterfaces:
                    logger.warning(f"network interfaces changed! new: {self.networkInterfaces} old: {self.jsonConfig['network']['interfaces']}")
                    self.jsonConfig['network']['interfaces'] = self.networkInterfaces
                    changed = True

                if self.jsonConfig['network']['preferredLinkSpeed'] != self.networkLinkSpeed:
                    logger.warning(f"network interfaces changed! new: {self.networkLinkSpeed} old: {self.jsonConfig['network']['preferredLinkSpeed']}")
                    self.jsonConfig['network']['preferredLinkSpeed'] = self.networkLinkSpeed
                    changed = True

                if changed:
                    with open(self.DEP_DANTE_CFG,"w") as outJsonFile:
                        json.dump(self.jsonConfig, outJsonFile, indent=4)
                        logger.info("Generated dante.json file")
                else:
                    logger.info("Nothing changed!")
        
        except Exception as e:
            logger.error(e)

if __name__ == "__main__":
    logging.basicConfig(filename='/var/log/dante-config-check.log', level=logging.INFO, format='%(asctime)s %(name)s %(levelname)s %(message)s')
    logger.info("Start dante-config application")
    danteCfg = DanteCfgGenerator()
    danteCfg.start()
