/* -----------------------------------------------------------------------------

notify.js
A simple notifications library built with pure javascript and no dependencies.
Copyright Drew Warkentin 2016. All rights reserved.

To do:
-Add a close button to close notifications => done
-Add notification show time => done
-Add support for non auto-dismissing notifications => done 
----------------------------------------------------------------------------- */

let notify = {
    // Some important variables
    NOTIFICATION_CLASS_NAME: "simple-notification",
    CONTAINER_CLASS_NAME: "simple-notification-container",
    CONTAINER_ID_NAME: "notificationContainer",
    MARGIN_BETWEEN_NOTIFICATIONS: 5, //px
    NOTIFICATION_WIDTH: 400, //px
    NOTIFICATION_TIME: 7000, //ms
    AUTO_DISMISSING: true,
    LEVEL: 'good',
    notificationCount: 0,

    notify: function () {
        // arguments
        this.options = arguments[0] || {};
        // custom dismissing
        notify.AUTO_DISMISSING = typeof this.options.autoDismissing == 'boolean' ? this.options.autoDismissing : true;
        // custom level
        notify.LEVEL = typeof this.options.level !== 'undefined' ? this.options.level : "good";
        // custom timeout
        notify.NOTIFICATION_TIME = (typeof this.options.notificationTime !== 'undefined' && typeof this.options.notificationTime == 'number') ? this.options.notificationTime : 7000;
        notify.notificationCount++;
        let notificationId = 'notification' + notify.notificationCount;
        let newNotification = {
            "id": notificationId,
            "message": this.options.message,
            "level": notify.LEVEL
        };
        // If we don't have the notification container already on the page, create it
        if (document.getElementById("notificationContainer")) {
            // Show the notification
            notify.showNotification(newNotification);
        } else {
            // create the container
            notify.createContainer();
            // Show the notification
            notify.showNotification(newNotification);
        }
    },

    // This function creates the container for the notifications to be render in
    createContainer: function () {
        let divContainer = document.createElement("div");
        divContainer.className = notify.CONTAINER_CLASS_NAME;
        divContainer.id = notify.CONTAINER_ID_NAME;
        document.body.appendChild(divContainer);
    },

    showNotification: function (notification) {
        // Create the new notification element
        let newNotification = document.createElement("div");
        newNotification.className = notify.NOTIFICATION_CLASS_NAME + " " + notification.level;
        newNotification.id = notification.id;
        newNotification.innerHTML = notification.message + "<div class='close-notification' onclick='notify.close(" + newNotification.id + ")'></div>";

        let isSafari = /^((?!chrome|android).)*safari/i.test(navigator.userAgent);
        if (!isSafari) {
            newNotification.style.marginLeft = notify.NOTIFICATION_WIDTH + 10 + "px";
        }
        // Create a wrapper for the notification element so that we can transition it nicely.
        let notificationWrapper = document.createElement("div");
        notificationWrapper.className = "simple-notification-wrapper";
        notificationWrapper.appendChild(newNotification);
        // Add it to the DOM
        let notificationContainer = document.getElementById(notify.CONTAINER_ID_NAME);
        notificationContainer.insertBefore(notificationWrapper, notificationContainer.firstChild);
        // if auto_dismissing equal true
        if (notify.AUTO_DISMISSING) {
            // Destroy the notification after the set time
            setTimeout(function () {
                notify.removeNotification(newNotification);
            }, notify.NOTIFICATION_TIME);
        }
    },

    close: function (notification) {
        notify.removeNotification(notification);
    },

    removeNotification: function (notificationToRemove) {
        notificationToRemove.className = notificationToRemove.className + " fade-out";
        // Remove the notification from the DOM after the fade out has finished
        notificationToRemove.addEventListener("transitionend", function () {
            document.getElementById(notify.CONTAINER_ID_NAME).removeChild(notificationToRemove.parentElement);
        }, false);
    }
}
