import os, logging

logger = logging.getLogger(__name__)

class Gpio:
    GPIO_ROOT = "/sys/class/gpio"
    GPIO_EXPORT = os.path.join(GPIO_ROOT, "export")
    gpiopath = lambda self, pin: os.path.join(self.GPIO_ROOT, "gpio{0}".format(pin))

    IN, OUT = "in", "out"

    def __init__(self, pin: str, mode: str, initial:bool=False, active_low:bool=True):
        if not isinstance(pin, str):
            raise TypeError(pin)
        self._pin = pin


        self._ppath = self.gpiopath(pin)
        if not os.path.exists(self._ppath):
            logger.debug("Creating GPIO {0}".format(pin))
            with open(self.GPIO_EXPORT, "w") as f:
                f.write(pin)

        self.activeLow  = active_low
        self.setDirection(mode)

        if mode == self.OUT:
            if initial:
                self.set(True)
            else:
                self.set(False)

    @property
    def fileno(self):
        return self._value.fileno()

    def read(self) -> int:
        self._value.seek(0)
        val = int(self._value.read().strip())
        #logger.debug("Read {0}: {1}".format(self._pin, val))
        return val

    def set(self, value: int):
        if self._mode == self.IN:
            return
        value = int(bool(value))
        logger.debug("Write {0}: {1}".format(self._pin, value))
        self._value.write(str(value))
        self._value.flush()

    def setDirection(self,mode: str) -> None:
        """
        Sets direction:
          * Gpio.IN or GPIO.OUT
        """
        if mode not in (self.IN, self.OUT):
            raise ValueError(mode)

        if mode==self._mode:
            return

        self._mode = mode

        if self._value is not None:
            self._value.close()

        with open(os.path.join(self._ppath, "direction"), "w+") as f:
            f.write(mode)

        if mode == self.OUT:
            fileMode = "w+"
            with open(os.path.join(self._ppath, "active_low"), "w+") as f:
                if self.active_low:
                    f.write('1')
                else:
                    f.write('0')
        else:
            fileMode = "r"
        
        try:
            self._value = open(os.path.join(self._ppath, "value"), fileMode)
        except Exception as e:
            if self._value:
                self._value.close()
                raise e

