#!/usr/bin/python3
import json, os, logging, sys, subprocess, platform, time
import logging.handlers
from smbus import SMBus

HW_FEATURES_FILE='/barix/info/DEVICES_FEATURES'
TMP_FEATURES_FILE='/tmp/device_features'
GPIO_API_PATH='/dev/gpio'
GPIO_SYS_PATH='/sys/class/gpio'
BM44_DRIVER_PATH='/sys/kernel/bm44'
PADAUK_ADDR1=0x42
PADAUK_ADDR2=0x43

#handler = logging.handlers.SysLogHandler(address='/dev/log')
#logformat = logging.Formatter('%(levelname)s %(filename)s:%(lineno)d %(message)s')
#handler.setFormatter(logformat)
#logging.basicConfig(handlers=[handler], level=logging.DEBUG)
logging.basicConfig(filename='/var/log/hw-api.log',
                    format='%(asctime)s %(levelname)s %(filename)s:%(lineno)d %(message)s',
                    datefmt='%m/%d/%Y %I:%M:%S %p',
                    level=logging.DEBUG)

logger = logging.getLogger('hw-api')


def readHwType() -> str:
    hwType = ''
    try:
        # qiba-spi-get-production-info.sh always return error
        hwType = subprocess.check_output('/usr/bin/qiba-spi-get-production-info.sh -w', shell=True, stderr=subprocess.STDOUT)
    except subprocess.CalledProcessError as e:
        hwType = str(e.output.decode('utf-8')).replace('\n', '')
    except Exception as err:
        logger.error("Error getting hw type: {}".format(err))
        raise err

    return hwType


def createDriverSymlink(ioType, idx, srcFile):
    logger.info(f'Creating symlinks: ioType {ioType} idx {idx} srcFile {srcFile}')
    if ioType == 'inputs':
        linkName = 'in'
    elif ioType == 'outputs':
        linkName = 'out'
    elif ioType == 'relays':
        linkName = 'rel'
    elif ioType == 'analogs':
        linkName = 'analog'
    elif ioType == 'pullups':
        linkName = 'pullup'
    else:
        logger.error(f'Unknown GPIO type: {ioType}')

    linkName = f'/dev/gpio/{linkName}{idx}'
    if not os.path.islink(linkName):
        logger.info(f'Creating symlink: {linkName} -> {srcFile}')
        os.symlink(srcFile, linkName)
    else:
        logger.debug(f'{linkName} already exists!')


def createBM44Api() -> dict:
    logger.info('Loading BM44 driver')
    logger.debug(f'OS Release: {platform.release()}')
    subprocess.run(f'insmod /lib/modules/{platform.release()}/extra/bm44.ko', shell=True,
            stderr=subprocess.DEVNULL)

    with open(f'{BM44_DRIVER_PATH}/info_io') as fp:
        lines = fp.readlines()

    ioInfo = {}
    for line in lines:
        fields = line.split()
        ioInfo[fields[0]] = int(fields[1])

    logger.debug(f'IO count for BM44: {ioInfo}')
    # create BM44 symlinks
    for ioType, count in ioInfo.items():
        for idx in range(1, ioInfo[ioType]+1):
            createDriverSymlink(ioType, idx, f'{BM44_DRIVER_PATH}/{ioType}/{idx}')

    return ioInfo

def createUX8Api() -> dict:
    with open(f'{BM44_DRIVER_PATH}/usb_enable', 'w') as fp:
        fp.write('1')

    # Discover all the UX8 devices
    discoverList = []
    timeout = time.monotonic() + 5.0
    while len(discoverList) < 4 and time.monotonic() <= timeout:
        for directory in os.listdir('/sys/kernel'):
            if directory.startswith('ux8-'):
                fields = directory.split('-')
                idx = int(fields[1])
                if idx not in discoverList:
                    with open(f'/sys/kernel/ux8-{idx}/info') as fp:
                        ux8Info = fp.read()
                    logger.info(f'Found UX8-{idx}: {ux8Info}')
                    discoverList.append(idx)

        if len(discoverList) < 4:
            time.sleep(0.1)

    # No UX8 was found. Disable USB.
    if len(discoverList) == 0:
        with open(f'{BM44_DRIVER_PATH}/usb_enable', 'w') as fp:
            fp.write('0')
        return {}

    ioCount = {}

    for ux8Idx in discoverList:
        with open(f'/sys/kernel/ux8-{ux8Idx}/info_io') as fp:
            lines = fp.readlines()

        ioInfo = {}
        for line in lines:
            fields = line.split()
            ioInfo[fields[0]] = int(fields[1])

        logger.debug(f'IO count for UX8-{ux8Idx}: {ioInfo}')
        # create symlinks
        for ioType, count in ioInfo.items():
            # fill dict to be returned in the end
            if ioType not in ioCount:
                ioCount[ioType] = count
            else:
                ioCount[ioType] += count
            # assuming BM44 has 4 IOs of each type and each UX8 has 8 IOs of each type
            firstIO = 5+(ux8Idx-1)*count
            lastIO = firstIO + count
            srcIdx = 1
            for idx in range(firstIO, lastIO):
                createDriverSymlink(ioType, idx, f'/sys/kernel/ux8-{ux8Idx}/{ioType}/{srcIdx}')
                srcIdx += 1

    return ioCount


# returns the number of IOs
def createGenericSymlinks(conf, gpioType) -> int:
    activeLow = '0'
    direction = 'low'
    if 'active_low' in conf and conf['active_low']:
        activeLow = '1'
        direction = 'high'
        logger.debug(f'Active low set to {activeLow} for all {gpioType}')

    if gpioType == 'inputs':
        direction = 'in'

    for idx, gpio in enumerate(conf['gpios']):
        if not os.path.exists(f'{GPIO_SYS_PATH}/gpio{gpio}'):
            with open(f'{GPIO_SYS_PATH}/export', 'w') as fp:
                fp.write(str(gpio))

        with open(f'{GPIO_SYS_PATH}/gpio{gpio}/active_low', 'w') as fp:
            fp.write(activeLow)

        with open(f'{GPIO_SYS_PATH}/gpio{gpio}/direction', 'w') as fp:
            fp.write(direction)

        if gpioType == 'relays':
            symlink = f'{GPIO_API_PATH}/rel{idx+1}'
        elif gpioType == 'inputs':
            symlink = f'{GPIO_API_PATH}/in{idx+1}'
        else:
            logger.error(f'Unknown GPIO type: {gpioType}')
            sys.exit(1)

        if not os.path.islink(symlink):
            logger.info(f'Creating symlink: {symlink} -> {GPIO_SYS_PATH}/gpio{gpio}')
            os.symlink(f'{GPIO_SYS_PATH}/gpio{gpio}', symlink)
        else:
            logger.debug(f'{symlink} already exists!')

    return len(conf['gpios'])

def getNotRunApps(applications: dict, hwType:int) -> list:
    notRunApps = []
    for context, apps in applications.items():
        for app, devices in apps.items():
            for dev in devices:
                if context == 'run_only' and dev != hwType:
                    notRunApps.append(app)
                elif context == 'not_run' and dev == hwType:
                    notRunApps.append(app)

    logger.info(f'Do not start these applications: {notRunApps}')
    return notRunApps

if __name__ == '__main__':
    try:
        with open(HW_FEATURES_FILE) as fp:
            hwDesc = json.load(fp)
    except Exception as exc:
        logger.error(f'Error reading file {HW_FEATURES_FILE}: {exc}')
        sys.exit(1)

    os.makedirs(GPIO_API_PATH, exist_ok=True)

    i2c = SMBus(0)
    padaukFound = False
    try:
        group = i2c.read_byte_data(PADAUK_ADDR1, 0x1c)
        board = i2c.read_byte_data(PADAUK_ADDR1, 0x1d)
        logger.debug(f'Bytes from Padauk: {group:02x} {board:02x}')
        padaukFound = True
    except Exception as exc:
        logger.info(f'Padauk not found: {exc}')

    boardEntry = None
    hwCapabilities = {}
    showIoDashboard = False
    showSnmpTraps = False

    # if padauk was found lets give it priority over hw type
    if padaukFound:
        logger.info('Padauk was found. Checking board bytes...')
        for entry in hwDesc['hw_features']:
            if 'padauk_bytes' in entry:
                if group == entry['padauk_bytes'][0] and \
                        board == entry['padauk_bytes'][1]:
                    boardEntry = entry
                    logger.info(f'Board found with padauk bytes: {hex(group)} {hex(board)}')

    # find board using hw type if not found on padauk bytes
    if boardEntry is None:
        # find hw type and create the symlinks
        hwType = readHwType()
        for entry in hwDesc['hw_features']:
            if entry['hw_type'] == hwType:
                boardEntry = entry
                break

        # if board not found, fill basic information
        if boardEntry is None:
            boardEntry = {}
            boardEntry['hw_type'] = hwType
            boardEntry['hw_type_descr'] = 'Unknown Board'

    if 'driver' in boardEntry and boardEntry['driver'] == 'bm44':
        showIoDashboard = True
        showSnmpTraps = True
        bm44Capab = createBM44Api()
        ux8Capab = createUX8Api()

        # sum BM44 capabilities and ux8 capabilites
        hwCapabilities = {io: bm44Capab.get(io, 0) + ux8Capab.get(io, 0)
                for io in set(bm44Capab) | set(ux8Capab)}

        # load W1 DS2482 driver for temperature sensor
        with open('/sys/bus/i2c/devices/i2c-0/new_device', 'w') as fp:
            fp.write('ds2482 0x18')

    else:
        logger.info(f'Creating GPIO symlinks for "{boardEntry["hw_type_descr"]}"')

        if 'relays' in boardEntry:
            count = createGenericSymlinks(boardEntry['relays'], 'relays')
            hwCapabilities['relays'] = count

        if 'inputs' in boardEntry:
            count = createGenericSymlinks(boardEntry['inputs'], 'inputs')
            hwCapabilities['inputs'] = count

    # create file under /tmp containing only the features for the found board
    featuresContent = {
            'hw_type': boardEntry['hw_type'],
            'hw_type_descr': boardEntry['hw_type_descr'],
            'hw_capabilities': hwCapabilities,
            'show_io_dashboard': showIoDashboard,
            'show_snmp_traps': showSnmpTraps
            }
    featuresContent['hidden_features'] = getNotRunApps(
            hwDesc['applications'], int(boardEntry['hw_type']))

    with open(TMP_FEATURES_FILE, 'w') as fp:
        json.dump(featuresContent, fp)
