import json
import os
import subprocess

from flask import Blueprint, Response, request, current_app

#from SdfConfigValuesValidator import SdfConfigValuesValidator
from SdfValueValidator import SdfValueValidator
from logger import get_backend_log
from middleware.endpoint_logger import endpoint_log

from barix.web.uci import getValueOfUci, setUciConfigs

web_bp = Blueprint('web_API', __name__)

logger = get_backend_log()


@web_bp.route('/sdf', methods=['GET'])
@endpoint_log
def get_sdf_file_content():
    sdf = current_app.config["sdf"]
    if sdf is None:
        return Response(status=500)
    if sdf == "SDF config not found":
        return Response(status=404)
    if not sdf.valid or sdf.content is None:
        return Response(status=400)
    return Response(json.dumps(sdf.content), headers={"Content-Type": "application/json"}, status=200)


@web_bp.route('/config', methods=['GET'])
@endpoint_log
def get_config():
    config = {}
    # check if config file exists
    if not os.path.isfile(current_app.config["values"]):
        logger.warn("App config file does not exist!")
    else:
        # read config file contents
        try:
            with open(current_app.config["values"], 'r') as f:
                config = json.load(f)
                if "AppParam" in config:
                    config = config["AppParam"]
        except Exception as e:
            logger.error("Error reading SDF JSON file: {}".format(e))
            return Response(status=500)

    return Response(json.dumps(config), headers={"Content-Type": "application/json"}, status=200)


@web_bp.route('/config', methods=['PUT'])
@endpoint_log
def set_config():
    data = request.data
    try:
        json_config = json.loads(data.decode('utf-8'))
    except Exception as e:
        logger.error("Error parsing request body: {}".format(e))
        return Response(status=400)

    try:
        # check if local config is enabled
        result = getValueOfUci('sdf', 'general', 'enable_local_conf')
        if result == "false":
            logger.error("Cannot store configuration because local config is disabled")
            return Response(status=403)

        # validate config received

        validator = SdfValueValidator(current_app.config["sdf"].content)

        errors = validator.validate(json_config)

        if len(errors) > 0:
            return Response(response=json.dumps(errors), status=400)

        # store configuration

        # if file doesn't exist
        if not os.path.isfile(current_app.config["values"]):
            with open(current_app.config["values"], 'w') as f:
                f.write(json.dumps({"AppParam": json_config}, indent=4))
        # if file exists:
        else:
            # read config file contents
            f = open(current_app.config["values"])
            config = json.load(f)
            f.close()
            config["AppParam"] = json_config
            with open(current_app.config["values"], 'w') as f:
                f.write(json.dumps(config, indent=4))

        # disable flexa agent
        setUciConfigs({"flexa_agent.service.enabled": 'false'}, commit=True,
                      restartServices=False,
                      selectServicesToRestartCallback=None)
        try:
            # restart the app
            subprocess.run("/etc/init.d/run-flexa restart", shell=True, timeout=10.0)
        except Exception as e:
            logger.error("Error restarting flexa application: {}".format(e))

        return Response(status=200)
    except Exception as e:
        logger.error("Error storing configuration: {}".format(e))
        return Response(status=500)


@web_bp.route('/system', methods=['GET'])
@endpoint_log
def get_system_state():
    try:
        uciValues = {}
        uciList = request.args

        for uciPath in uciList:
            path_params = uciPath.split('.')
            result = getValueOfUci(path_params[0], path_params[1], path_params[2])
            uciValues[uciPath] = result
    except Exception as e:
        logger.error(e)
        return Response(status=500)
    else:
        return Response(json.dumps(uciValues), status=200)