from typing import List, Dict, Any
import re
import json

from logger import get_backend_log
logger = get_backend_log()


class ScopeEntry:
    def __init__(self, key: str, value: Any):
        self.key = key
        self.value = value


class SdfValueValidator:
    PARAM_TYPE = 'param'
    SECTION_TYPE = 'section'
    DIVISION_TYPE = 'division'

    def __init__(self, sdf_json: Dict[str, Any]):
        self.sdf = sdf_json
        self.scopes_catalog: Dict[str, List[ScopeEntry]] = {}
        self.DEBUG = False

    def validate(self, json_value_to_validate: Any) -> List[str]:
        errors = []
        # if self.DEBUG is True:
        #     logger.debug(f"==> VALIDATE ----------------------------------------")
        #     logger.debug(f"==> JSON TO VALIDATE {json.dumps(json_value_to_validate)}")
        #     logger.debug(f"==> self.sdf['config'] {json.dumps(self.sdf['config'])}")
        errors += self.validate_config(json_value_to_validate, self.sdf['config'])
        errors += self.validate_scopes()
        return errors

    def validate_config(self, values: Dict[str, Any], sdf_config: List[Dict[str, Any]]) -> List[str]:
        errors = []
        index = 0
        for conf in sdf_config:
            c_type = conf.get('c_type')
            if c_type == self.PARAM_TYPE:
                if self.DEBUG is True:
                    logger.debug(f"==> VALIDATING PARAM \"{conf['c_data']['key']}\" ----------------------------------------")
                param_type_errors = self.validate_param(values.get(conf['c_data']['key']), values, conf['c_data'])
                if self.DEBUG is True:
                    logger.debug(f"--- Param type errors: {param_type_errors}")
                errors.extend(param_type_errors)
            elif c_type == self.SECTION_TYPE:
                if self.DEBUG is True:
                    logger.debug(f"==> VALIDATING SECTION \"{conf['c_data']['key']}\" ----------------------------------------")
                errors.extend(self.validate_section(values.get(conf['c_data']['key']), values, conf['c_data']))
            index += 1
        return errors

    def validate_param(self, value: Any, values: Dict[str, Any], sdf_data: Dict[str, Any]) -> List[str]:
        errors = []

        if self.DEBUG is True:
            logger.debug(f"--- required: {sdf_data.get('required')}, value: \"{value}\"")

        if value is None:
            if sdf_data.get('required'):
                return [f"Attribute '{sdf_data['key']}' is required."]
            else:
                return []

        self.add_param_to_scope(value, sdf_data)

        var_type = sdf_data.get('var_type')
        if self.DEBUG is True:
            logger.debug(f"--- var_type: {var_type}")

        if var_type:
            if var_type == 'boolean':
                if not isinstance(value, bool):
                    errors.append(f"Attribute '{sdf_data['key']}' type mismatch. Must be a '{var_type}'.")
            elif var_type == 'number':
                if not isinstance(value, (int, float)):
                    errors.append(f"Attribute '{sdf_data['key']}' type mismatch. Must be a '{var_type}'.")
                else:
                    num_err_bound = self.validate_number_boundaries(value, sdf_data)
                    errors.extend(num_err_bound)
            elif var_type == 'string':
                if not isinstance(value, str):
                    errors.append(f"Attribute '{sdf_data['key']}' type mismatch. Must be a '{var_type}'.")
                else:
                    if not self.is_empty_value(value):
                        regex_err = self.validate_string_regexp(value, sdf_data)
                        errors.extend(regex_err)
            elif var_type == 'number[]':
                if not isinstance(value, list) or (value and not all(isinstance(x, (int, float)) for x in value)):
                    errors.append(f"Attribute '{sdf_data['key']}' type mismatch. Must be a '{var_type}'.")
                else:
                    for val in value:
                        num_err = self.validate_number_boundaries(val, sdf_data)
                        errors.extend(num_err)
            elif var_type == 'string[]':
                if not isinstance(value, list) or (value and not all(isinstance(x, str) for x in value)):
                    errors.append(f"Attribute '{sdf_data['key']}' type mismatch. Must be a '{var_type}'.")
                else:
                    for val in value:
                        regex_err = self.validate_string_regexp(val, sdf_data)
                        errors.extend(regex_err)

        if sdf_data.get('conditional'):
            cond_err = self.validate_conditionals(sdf_data['conditional'], value, values)
            errors.extend(cond_err)

        return errors

    @staticmethod
    def validate_string_regexp(value: Any, sdf_data: Dict[str, Any]) -> List[str]:
        if value is not None and sdf_data.get('type_properties') and sdf_data['type_properties'].get('regexp'):
            the_regex = sdf_data['type_properties']['regexp']
            regext_match = re.search(the_regex, value)
            if not isinstance(value, str) or not regext_match:
                return [f"Attribute '{sdf_data['key']}' doesn't match format '{the_regex}'."]
        return []

    @staticmethod
    def validate_number_boundaries(value: Any, sdf_data: Dict[str, Any]) -> List[str]:
        errors = []
        if value is not None:
            min_val = sdf_data['type_properties'].get('min')
            max_val = sdf_data['type_properties'].get('max')
            if min_val is not None and int(value) < int(min_val):
                errors.append(f"Attribute '{sdf_data['key']}' can not be less than '{min_val}'.")
            if max_val is not None and int(value) > int(max_val):
                errors.append(f"Attribute '{sdf_data['key']}' can not be greater than '{max_val}'.")
        return errors

    def validate_section(self, value: Any, values: Dict[str, Any], sdf_data: Dict[str, Any]) -> List[str]:
        # if self.DEBUG is True:
        #     logger.debug(f"==> VALIDATING SECTION ----------------------------------------")
        #     logger.debug(f"==> VALUE {json.dumps(value)}")
        #     logger.debug(f"==> VALUES {json.dumps(values)}")
        #     logger.debug(f"==> SDF_DATA {json.dumps(sdf_data)}")
        errors = []
        if value is not None and sdf_data.get('content'):
            errors = self.validate_config(value, sdf_data.get('content'))
        return errors

    def validate_conditionals(self, conditional: List[Dict[str, Any]], value: Any, values: Dict[str, Any]) -> List[str]:
        if not conditional:
            return []

        errors = []
        if self.DEBUG is True:
            logger.debug(f"--- validating conditionals: {conditional}")
        for cond in conditional:
            active_configs = []

            for cond_key, cond_value in cond.items():
                is_array_validation = False
                comparator_fn = lambda x, y: False

                if cond_key == 'when_value_is':
                    is_array_validation = True
                    comparator_fn = lambda cond_entry, val: all(x in val for x in cond_entry) if isinstance(cond_entry, list) else str(cond_entry) == str(val)
                elif cond_key == 'when_value_is_not':
                    is_array_validation = True
                    comparator_fn = lambda cond_entry, val: not all(x in val for x in cond_entry) if isinstance(cond_entry, list) else str(cond_entry) != str(val)
                elif cond_key == 'when_value_starts_with':
                    is_array_validation = True
                    comparator_fn = lambda cond_entry, val: str(val).startswith(str(cond_entry))
                elif cond_key == 'when_value_includes':
                    is_array_validation = True
                    comparator_fn = lambda cond_entry, val: all(x in val for x in cond_entry) if isinstance(cond_entry, list) else str(cond_entry) in str(val)
                elif cond_key == 'when_value_matches_regex':
                    is_array_validation = True
                    comparator_fn = lambda cond_entry, val: bool(re.match(str(cond_entry), str(val)))
                elif cond_key.startswith('when_value_is_'):
                    is_array_validation = False
                    operator = cond_key[len('when_value_is_'):]
                    comparator_fn = getattr(self, f'_compare_{operator}', lambda x, y: False)

                if cond_value is not None and isinstance(cond_value, list):
                    match = any(comparator_fn(cond_entry, value) for cond_entry in cond_value) if is_array_validation else comparator_fn(cond_value, value)
                    if match:
                        active_configs.extend(cond['config'])
                        break

            for active_config in active_configs:
                errors.extend(self.validate_config(values, [active_config] ))

        return errors

    def add_param_to_scope(self, value: Any, sdf_data: Dict[str, Any]) -> None:
        if sdf_data.get('scope') and sdf_data['scope'] in [sn['name'] for sn in self.sdf.get('scopes', [])]:
            entry = ScopeEntry(sdf_data['key'], value)
            self.scopes_catalog.setdefault(sdf_data['scope'], []).append(entry)

    def validate_scopes(self) -> List[str]:
        errors = []
        if 'scopes' in self.sdf:
            for key, scope_values in self.scopes_catalog.items():
                scope_def_index = next((i for i, sd in enumerate(self.sdf.get('scopes', [])) if sd['name'] == key), -1)

                if scope_def_index > -1:
                    scope_definition = self.sdf['scopes'][scope_def_index]

                    if scope_definition.get('max_occurrences') and scope_values:
                        scope_counter = {}
                        for entry in scope_values:
                            scope_counter[entry.value] = scope_counter.get(entry.value, 0) + 1

                        if all(value > scope_definition['max_occurrences'] for value in scope_counter.values()):
                            errors.append(f"Attributes {[k.key for k in scope_values]} values are invalid. "
                                          f"Scope '{key}' does not allow more than {scope_definition['max_occurrences']} occurrences.")

                    # min_occurrences
                    # ...

        return errors

    @staticmethod
    def is_empty_value(value: Any) -> bool:
        return (
            value is None or
            ((isinstance(value, str) or isinstance(value, list)) and len(value) == 0)
        )

    @staticmethod
    def _array_contains_all_elements(arr1: List[Any], arr2: List[Any]) -> bool:
        return all(element in arr1 for element in arr2)
