#!/bin/sh

# Barix ExstreamerL Board Specific Setup and Utilities.
# Copyright (C) 2018 Barix AG (http://www.barix.com). All rights reserved.

# Testing on host (instead of target) specific features
# Allowed to be specified externally.
# Overrides LIBPATH and SYSPATH.
[ -z "${TEST_HOST}" ] && TEST_HOST="0"
if [ "${TEST_HOST}" = "1" ]; then
	LIBPATH="."
	SYSPATH="./test"
fi

# The scripts path root.
# Allowed to be specified externally, e.g., for testing.
[ -z "${LIBPATH}" ] && LIBPATH="/usr/bin"

# The sys path root.
# Allowed to be specified externally, e.g., for testing.
[ -z "${SYSPATH}" ] && SYSPATH=""

# The log messages prefix.
# Allowed to be specified externally.
[ -z "${LOG_PFX}" ] && LOG_PFX="qiba-exstreamerl"

# The list of the I2C GPIO Expanders' addresses and busses.
readonly GPIOEXP_I2C_ADDR=(0x20 0x21)
readonly GPIOEXP_I2C_BUS=(0 0)

# The list of signals in I2C GPIO Expander 0.
# IMPORTANT: The signal names must be unique between all devices.
readonly GPIOEXP_0_SIG=( \
	"LED_1" "LED_2" "LED_3" "LED_4" "LED_5"  "LED_6"  "LED_7"  "LED_8"  \
	"INL_H" "INR_H" "BNC_I" "TP_25" "OUTL_I" "OUTL_H" "OUTR_I" "OUTR_H" \
)
# The list of GPIO directions of I2C GPIO Expander 0.
readonly GPIOEXP_0_DIR=( \
	"out"   "out"   "out"   "out"   "out"    "out"    "out"    "out"    \
	"out"   "out"   "out"   "out"   "out"    "out"    "out"    "out"    \
)
# The list of GPIO reset values of I2C GPIO Expander 0 (outputs only).
readonly GPIOEXP_0_VAL=( \
	"0"     "0"     "0"     "0"     "0"      "0"      "0"      "0"      \
	"0"     "0"     "0"     "0"     "0"      "0"      "0"      "0"      \
)

# The list of signals in I2C GPIO Expander 1.
# IMPORTANT: The signal names must be unique between all devices.
readonly GPIOEXP_1_SIG=( \
	"TP_24" "TP_19" "TP_20" "TP_21" "LED_2R" "LED_2G" "LED_1R" "LED_1G" \
	"INP_1" "INP_2" "INP_3" "INP_4" "OUTP_1" "OUTP_2" "OUTP_3" "OUTP_4" \
)
# The list of GPIO directions of I2C GPIO Expander 1.
readonly GPIOEXP_1_DIR=( \
	"out"   "out"   "out"   "out"   "out"    "out"    "out"    "out"    \
	"in"    "in"    "in"    "in"    "out"    "out"    "out"    "out"    \
)
# The list of GPIO reset values of I2C GPIO Expander 1 (outputs only).
readonly GPIOEXP_1_VAL=( \
	"0"     "0"     "0"     "0"     "1"      "0"      "1"      "0"      \
	"0"     "0"     "0"     "0"     "0"      "0"      "0"      "0"      \
)

# Number of I2C GPIO Expanders.
# Each will have an entry in busses and addresses lists.
readonly GPIOEXP_NUM=${#GPIOEXP_I2C_ADDR[@]}

# File that marks initialized state.
readonly GPIOEXP_FILE_INIT="/tmp/.qiba-gpioexp-init"

# The SPI bus number of the SPI master controlling the amplifiers:
readonly SPIAMP_BUS="2"

# SPI amplifier indexes:
#  + 0: Input Left.
#  + 1: Input Right.
#  + 2: Output Left.
#  + 3: Output Right.
readonly SPIAMP_NAME=("inl" "inr" "outl" "outr")

# SPI amplifier index: Minimum.
readonly SPIAMP_IDX_MIN="0"
# SPI amplifier index: Maximum.
readonly SPIAMP_IDX_MAX="$((${#SPIAMP_NAME[@]} - 1))"
# SPI amplifier gain: Minimum.
readonly SPIAMP_GAIN_MIN="0"
# SPI amplifier gain: Maximum without 'force'.
readonly SPIAMP_GAIN_MAX="18"
# SPI amplifier gain: Maximum with 'force'.
readonly SPIAMP_GAIN_MAX_FORCE="60"

# Logging facilities.
source ${LIBPATH}/qiba-log.sh
# GPIO utilities.
source ${LIBPATH}/qiba-gpio.sh
# I2C GPIO Expander utilities.
source ${LIBPATH}/qiba-i2c-gpio.sh

# GPIO Expander: Reset GPIOs to their registered values (direction, value,
# etc.).
#
# Parameters:
#  $1: The GPIO Expander device ID.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
_gpioexp_dev_reset() {
	local dev_idx=${1}

	local bus addr sigs dirs vals sig dir val msg

	local ret=0
	local retf=0
	local f="${FUNCNAME[0]}"

	# Get the device's I2C bus and address.
	bus=${GPIOEXP_I2C_BUS[dev_idx]}
	addr=${GPIOEXP_I2C_ADDR[dev_idx]}

	# Get the list of signals, directions and values for the current device.
	# Variables 'sigs', 'dirs' and 'vals' acting as kind of pointers.
	eval sigs="\(\${GPIOEXP_${dev_idx}_SIG[@]}\)"
	eval sigs="${sigs}"
	eval dirs="\(\${GPIOEXP_${dev_idx}_DIR[@]}\)"
	eval dirs="${dirs}"
	eval vals="\(\${GPIOEXP_${dev_idx}_VAL[@]}\)"
	eval vals="${vals}"

	# Set the direction on each of the GPIO Expander device's GPIO.
	for gpio in $(seq 0 $((${#sigs[@]} - 1))); do
		ret=0
		sig=${sigs[gpio]}
		dir=${dirs[gpio]}
		val=${vals[gpio]}
		i2c_gpio_dir_set "${bus}" "${addr}" "${dir}" "${gpio}" \
			|| ret=$?
		if [ "${dir}" = "out" ] || [ "${TEST_HOST}" = "1" ]; then
			i2c_gpio_val_set "${bus}" "${addr}" "${val}" "${gpio}" \
				|| ret=$?
		fi
		msg=$(printf "I2C [%s,%s] GPIO %2d / %6s: Set as %5s" \
			${bus} ${addr} ${gpio} ${sig} "'${dir}'")
		if [ ${ret} -ne 0 ]; then
			if [ ${retf} -eq 0 ]; then
				retf=${ret}
			fi
			log_inf "${f}" "${msg} FAILED (${ret})"
			continue
		fi
		if [ "${dir}" = "out" ] || [ "${TEST_HOST}" = "1" ]; then
			log_inf "${f}" "${msg} OK (val=${val})"
		else
			log_inf "${f}" "${msg} OK"
		fi
	done

	return ${retf}
}

# GPIO Expander: Initialize a device.
#
# Parameters:
#  $1: The GPIO Expander device ID.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
_gpioexp_dev_init() {
	local dev_idx=${1}

	local bus addr

	local ret=0
	local f="${FUNCNAME[0]}"

	# Export the GPIOs.
	bus=${GPIOEXP_I2C_BUS[dev_idx]}
	addr=${GPIOEXP_I2C_ADDR[dev_idx]}
	i2c_gpio_export "${bus}" "${addr}" || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "I2C [${bus},${addr}] GPIOs: Export FAILED (${ret})"
	else
		log_inf "${f}" "I2C [${bus},${addr}] GPIOs: Export OK"
	fi

	# Reset GPIOs.
	_gpioexp_dev_reset "${dev_idx}" || ret=$?

	return ${ret}
}

# GPIO Expander: Terminate a device.
#
# Parameters:
#  $1: The GPIO Expander device ID.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
_gpioexp_dev_term() {
	local dev_idx=${1}

	local bus addr

	local ret=0
	local f="${FUNCNAME[0]}"

	# Export the GPIOs.
	bus=${GPIOEXP_I2C_BUS[dev_idx]}
	addr=${GPIOEXP_I2C_ADDR[dev_idx]}
	i2c_gpio_unexport "${bus}" "${addr}" || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "I2C [${bus},${addr}] GPIOs: Unexport FAILED (${ret})"
		return ${ret}
	fi
	log_inf "${f}" "I2C [${bus},${addr}] GPIOs: Unexport OK"

	return 0
}

# GPIO Expander: Find a signal by name and get its data.
#
# Parameters:
#  $1: The signal name.
#  $2: [opt] Variable to store the signal's I2C bus number.
#  $3: [opt] Variable to store the signal's I2C address.
#  $4: [opt] Variable to store the signal's direction.
#  $5: [opt] Variable to store the signal's GPIO number.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
#  <outp_*>: The signal's data elements.
_gpioexp_sig_find() {
	local sig=${1}
	local outp_bus=${2}
	local outp_addr=${3}
	local outp_dir=${4}
	local outp_gpio=${5}

	local _dev_idx _bus _addr sigs dirs _gpio _sig _dir

	local f="${FUNCNAME[0]}"

	for _dev_idx in $(seq 0 $((${GPIOEXP_NUM} - 1))); do
		# Get I2C bus number and address.
		_bus=${GPIOEXP_I2C_BUS[_dev_idx]}
		_addr=${GPIOEXP_I2C_ADDR[_dev_idx]}
		# Get the list of signals and directions for the current device.
		# Variables 'sigs' and 'dirs' acting as kind of pointers.
		eval sigs="\(\${GPIOEXP_${_dev_idx}_SIG[@]}\)"
		eval sigs="${sigs}"
		eval dirs="\(\${GPIOEXP_${_dev_idx}_DIR[@]}\)"
		eval dirs="${dirs}"
		# Get each signal's data.
		for _gpio in $(seq 0 $((${#sigs[@]} - 1))); do
			_sig=${sigs[_gpio]}
			_dir=${dirs[_gpio]}
			if [ "${sig}" = "${_sig}" ]; then
				# Signal found => Set output params.
				[ -n "${outp_bus}" ] && eval ${outp_bus}="'${_bus}'"
				[ -n "${outp_addr}" ] && eval ${outp_addr}="'${_addr}'"
				[ -n "${outp_dir}" ] && eval ${outp_dir}="'${_dir}'"
				[ -n "${outp_gpio}" ] && eval ${outp_gpio}="'${_gpio}'"
				return 0
			fi
		done
	done

	return 1
}

# GPIO Expander: Show available signals and their parameters / values.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpioexp_show() {
	local dev_idx bus addr sigs dirs gpio sig dir sep1 sep2 val

	local f="${FUNCNAME[0]}"

	sep1="+-------------------------------+"
	sep2="+--------+-------+------+-------+"
	for dev_idx in $(seq 0 $((${GPIOEXP_NUM} - 1))); do
		# Get I2C bus number and address.
		bus=${GPIOEXP_I2C_BUS[dev_idx]}
		addr=${GPIOEXP_I2C_ADDR[dev_idx]}
		log_inf "${f}" "${sep1}"
		line=$(printf "| I2C Bus %s, Address 0x%03x      |" ${bus} ${addr})
		log_inf "${f}" "${line}"
		log_inf "${f}" "${sep2}"
		log_inf "${f}" "| SIGNAL |  DIR  | GPIO | VALUE |"
		log_inf "${f}" "${sep2}"
		# Get the list of signals and directions for the current device.
		# Variables 'sigs' and 'dirs' acting as kind of pointers.
		eval sigs="\(\${GPIOEXP_${dev_idx}_SIG[@]}\)"
		eval sigs="${sigs}"
		eval dirs="\(\${GPIOEXP_${dev_idx}_DIR[@]}\)"
		eval dirs="${dirs}"
		# Get each signal's data.
		for gpio in $(seq 0 $((${#sigs[@]} - 1))); do
			sig=${sigs[gpio]}
			dir=${dirs[gpio]}
			# Get the value (if initialized), otherwise mark value as
			# uninitialized.
			val="U"
			if [ -f ${GPIOEXP_FILE_INIT} ]; then
				i2c_gpio_val_get "${bus}" "${addr}" "${gpio}" val
			fi
			line=$(printf "| %6s |  %3s  |  %2s  |   %s   |" \
				${sig} ${dir} ${gpio} ${val})
			log_inf "${f}" "${line}"
		done
	done
	log_inf "${f}" "${sep2}"

	return 0
}

# GPIO Expander: Initialize all registered GPIO Expander devices.
#
# Exports the GPIOs and sets their registered directions, thus allowing the
# associated GPIOs to be manipulated (set, read, etc.).
#
# The registered devices' information given by:
#  + GPIOEXP_I2C_ADDR[]: Devices' I2C addresses.
#  + GPIOEXP_I2C_BUS[]: Devices' I2C bus numbers.
#  + GPIOEXP_<N>_SIG[]: Devices' GPIO signal names (<X> is device ID).
#  + GPIOEXP_<N>_DIR[]: Devices' GPIO signal directions (<X> is device ID).
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpioexp_init() {
	local dev_idx

	local ret=0
	local retf=0
	local f="${FUNCNAME[0]}"

	# Initialize only once.
	if [ -f ${GPIOEXP_FILE_INIT} ]; then
		log_inf "${f}" "GPIO Expanders Init OK (already initialized)"
		return 0
	fi

	# Initialize each I2C GPIO Expander device.
	for dev_idx in $(seq 0 $((${GPIOEXP_NUM} - 1))); do
		ret=0
		_gpioexp_dev_init "${dev_idx}" || ret=$?
		if [ ${ret} -ne 0 ] && [ ${retf} -eq 0 ]; then
			retf=${ret}
		fi
	done

	if [ ${retf} -eq 0 ]; then
		log_inf "${f}" "GPIO Expanders Init OK"
	else
		log_err "${f}" "GPIO Expanders Init FAILED (${retf})"
	fi

	# Mark initialized state.
	echo "" > ${GPIOEXP_FILE_INIT}

	return ${retf}
}

# GPIO Expander: Terminate all registered GPIO Expander devices.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpioexp_term() {
	local dev_idx

	local ret=0
	local retf=0
	local f="${FUNCNAME[0]}"

	# Terminate only once.
	if [ ! -f ${GPIOEXP_FILE_INIT} ]; then
		log_inf "${f}" "GPIO Expanders Term OK (not initialized)"
		return 0
	fi

	# Terminate each I2C GPIO Expander device.
	for dev_idx in $(seq 0 $((${GPIOEXP_NUM} - 1))); do
		ret=0
		_gpioexp_dev_term "${dev_idx}" || ret=$?
		if [ ${ret} -ne 0 ] && [ ${retf} -eq 0 ]; then
			retf=${ret}
		fi
	done

	if [ ${retf} -eq 0 ]; then
		log_inf "${f}" "GPIO Expanders Term OK"
	else
		log_err "${f}" "GPIO Expanders Term FAILED (${retf})"
	fi
	# Remove initialized state.
	rm -f ${GPIOEXP_FILE_INIT}

	return ${retf}
}

# GPIO Expander: Set the value of a signal.
#
# Parameters:
#  $1: The signal name.
#  $2: The signal value.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpioexp_sig_set() {
	local sig=${1}
	local val=${2}

	local bus addr dir gpio

	local ret=0
	local f="${FUNCNAME[0]}"

	if [ ! -f ${GPIOEXP_FILE_INIT} ]; then
		log_inf "${f}" "GPIO Expanders not initialized"
		return 1
	fi
	if [ -z "${sig}" ]; then
		log_err "${f}" "Missing signal name parameter"
		return 1
	fi
	if [ -z "${val}" ]; then
		log_err "${f}" "Missing signal value parameter"
		return 1
	fi

	# Find the signal and get its data.
	_gpioexp_sig_find "${sig}" bus addr dir gpio || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't find the '${sig}' signal {${ret}}"
		return ${ret}
	fi

	# Set only allowed on output signals.
	if [ "${dir}" != "out" ]; then
		log_err "${f}" "Signal '${sig}' is not output"
		return 1
	fi

	# Set the signal's GPIO value.
	i2c_gpio_val_set "${bus}" "${addr}" "${val}" "${gpio}" || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" \
			"Set '${sig}' [${bus},${addr},${gpio}] to ${val}: FAILED (${ret})"
		return ${ret}
	fi

	log_inf "${f}" "Set '${sig}' [${bus},${addr},${gpio}] to ${val}: OK"
	return 0
}

# GPIO Expander: Get the value of a signal.
#
# Parameters:
#  $1: The signal name.
#  $2: [opt] Variable to store the signal's value.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
#  <echo>|<outp>: The signal's value.
gpioexp_sig_get() {
	local sig=${1}
	local outp=${2}

	local bus addr dir gpio val

	local ret=0
	local f="${FUNCNAME[0]}"

	if [ ! -f ${GPIOEXP_FILE_INIT} ]; then
		log_inf "${f}" "GPIO Expanders not initialized"
		return 1
	fi
	if [ -z "${sig}" ]; then
		log_err "${f}" "Missing signal name parameter"
		return 1
	fi

	# Find the signal and get its data.
	_gpioexp_sig_find "${sig}" bus addr dir gpio || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't find the '${sig}' signal {${ret}}"
		return ${ret}
	fi

	# Set the signal's GPIO value.
	i2c_gpio_val_get "${bus}" "${addr}" "${gpio}" val || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Get '${sig}' [${bus},${addr},${gpio}]: FAILED (${ret})"
		return ${ret}
	fi
	log_inf "${f}" "Get '${sig}' [${bus},${addr},${gpio}]: OK"

	# If there is a variable to store the output, use it. Otherwise just
	# return the output through echoing.
	[ -n "${outp}" ] && eval ${outp}="'${val}'" || echo "value = [${val}]"

	return 0
}

# GPIO Expander: Audio Inputs: Connect to IPAM400.
#
# Connects audio inputs to IPAM400 and not Hiband.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpioexp_audio_in_ipam400() {
	local ret=0
	local f="${FUNCNAME[0]}"

	[ ${ret} -eq 0 ] && gpioexp_sig_set "INL_H" "0" || ret=$?
	[ ${ret} -eq 0 ] && gpioexp_sig_set "INR_H" "0" || ret=$?

	if [ ${ret} -eq 0 ]; then
		log_inf "${f}" "Audio Inputs to IPAM400 OK"
	else
		log_err "${f}" "Audio Inputs to IPAM400 FAILED (${ret})"
	fi

	return ${ret}
}

# GPIO Expander: Audio Inputs: Connect to Hiband.
#
# Connects audio inputs to Hiband and not IPAM400.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpioexp_audio_in_hiband() {
	local ret=0
	local f="${FUNCNAME[0]}"

	[ ${ret} -eq 0 ] && gpioexp_sig_set "INL_H" "1" || ret=$?
	[ ${ret} -eq 0 ] && gpioexp_sig_set "INR_H" "1" || ret=$?

	if [ ${ret} -eq 0 ]; then
		log_inf "${f}" "Audio Inputs to Hiband OK"
	else
		log_err "${f}" "Audio Inputs to Hiband FAILED (${ret})"
	fi

	return ${ret}
}

# GPIO Expander: Audio Outputs: Mute.
#
# Disconnects audio outputs from both IPAM400 and Hiband (mute).
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpioexp_audio_out_mute() {
	local ret=0
	local f="${FUNCNAME[0]}"

	[ ${ret} -eq 0 ] && gpioexp_sig_set "OUTL_I" "0" || ret=$?
	[ ${ret} -eq 0 ] && gpioexp_sig_set "OUTL_H" "0" || ret=$?
	[ ${ret} -eq 0 ] && gpioexp_sig_set "OUTR_I" "0" || ret=$?
	[ ${ret} -eq 0 ] && gpioexp_sig_set "OUTR_H" "0" || ret=$?

	if [ ${ret} -eq 0 ]; then
		log_inf "${f}" "Audio Outputs Mute OK"
	else
		log_err "${f}" "Audio Outputs Mute FAILED (${ret})"
	fi

	return ${ret}
}

# GPIO Expander: Audio Outputs: Connect to IPAM400.
#
# Connects audio outputs to IPAM400 and not Hiband.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpioexp_audio_out_ipam400() {
	local ret=0
	local f="${FUNCNAME[0]}"

	[ ${ret} -eq 0 ] && gpioexp_sig_set "OUTL_I" "1" || ret=$?
	[ ${ret} -eq 0 ] && gpioexp_sig_set "OUTL_H" "0" || ret=$?
	[ ${ret} -eq 0 ] && gpioexp_sig_set "OUTR_I" "1" || ret=$?
	[ ${ret} -eq 0 ] && gpioexp_sig_set "OUTR_H" "0" || ret=$?

	if [ ${ret} -eq 0 ]; then
		log_inf "${f}" "Audio Outputs from IPAM400 OK"
	else
		log_err "${f}" "Audio Outputs from IPAM400 FAILED (${ret})"
	fi

	return ${ret}
}

# GPIO Expander: Audio Outputs: Connect to Hiband.
#
# Connects audio outputs to Hiband and not IPAM400.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpioexp_audio_out_hiband() {
	local ret=0
	local f="${FUNCNAME[0]}"

	[ ${ret} -eq 0 ] && gpioexp_sig_set "OUTL_I" "0" || ret=$?
	[ ${ret} -eq 0 ] && gpioexp_sig_set "OUTL_H" "1" || ret=$?
	[ ${ret} -eq 0 ] && gpioexp_sig_set "OUTR_I" "0" || ret=$?
	[ ${ret} -eq 0 ] && gpioexp_sig_set "OUTR_H" "1" || ret=$?

	if [ ${ret} -eq 0 ]; then
		log_inf "${f}" "Audio Outputs from Hiband OK"
	else
		log_err "${f}" "Audio Outputs from Hiband FAILED (${ret})"
	fi

	return ${ret}
}

# GPIO Expander: Audio: Connect to IPAM400.
#
# Connects audio inputs/outputs to/from IPAM400 and not Hiband.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpioexp_audio_ipam400() {
	local ret=0
	local f="${FUNCNAME[0]}"

	[ ${ret} -eq 0 ] && gpioexp_audio_in_ipam400 || ret=$?
	[ ${ret} -eq 0 ] && gpioexp_audio_out_ipam400 || ret=$?

	if [ ${ret} -eq 0 ]; then
		log_inf "${f}" "Audio to/from IPAM400 OK"
	else
		log_err "${f}" "Audio to/from IPAM400 FAILED (${ret})"
	fi

	return ${ret}
}

# GPIO Expander: Audio: Connect to Hiband.
#
# Connects audio inputs/outputs to/from Hiband and not IPAM400.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpioexp_audio_hiband() {
	local ret=0
	local f="${FUNCNAME[0]}"

	[ ${ret} -eq 0 ] && gpioexp_audio_in_hiband || ret=$?
	[ ${ret} -eq 0 ] && gpioexp_audio_out_hiband || ret=$?

	if [ ${ret} -eq 0 ]; then
		log_inf "${f}" "Audio to/from Hiband OK"
	else
		log_err "${f}" "Audio to/from Hiband FAILED (${ret})"
	fi

	return ${ret}
}

# SPI Amplifiers: Find an amplifier index by name.
#
# Parameters:
#  $1: The amplifier name, one of SPIAMP_NAME[] entries.
#  $2: [opt] Variable to store the amplifier's index.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
#  <outp>: The amplifier's index.
_spiamp_find() {
	local name=${1}
	local outp=${2}

	local _idx _name

	local f="${FUNCNAME[0]}"

	for _idx in $(seq ${SPIAMP_IDX_MIN} ${SPIAMP_IDX_MAX}); do
		_name=${SPIAMP_NAME[_idx]}
		if [ "${name}" = "${_name}" ]; then
			# Amplifier found => Set output params.
			[ -n "${outp}" ] && eval ${outp}="'${_idx}'"
			return 0
		fi
	done

	return 1
}

# SPI Amplifiers: Set gain on amplifier (by index).
#
# Parameters:
#  $1: The SPI amplifier index, matching the SPI chip select.
#  $2: The gain value in dB. Valid values are from 0 to 60, in 3dB steps.
#   Anything above 18 requires $3 set to 'force', otherwise rejected.
#   The amplifiers only accept gain in 3dB multiples, so any value specified
#   that is not a multiple is rounded down to the previous multiple.
#  $3: [opt] Force indicator, required to be set to 'force' for setting gain
#   values larger than 18dB.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
_spiamp_gain_set() {
	local idx="${1}"
	local gain="${2}"
	local force="${3}"

	local min max gain_val flags spi_cmd spi_arg

	local ret=0
	local f="${FUNCNAME[0]}"

	# Confirm index and gain are specified as a decimal integers.
	if ! [ "${idx}" -eq "${idx}" ] 2>/dev/null; then
		log_err "${f}" "Missing or invalid amplifier index parameter '${idx}'"
		return 1
	fi
	if ! [ "${gain}" -eq "${gain}" ] 2>/dev/null; then
		log_err "${f}" "Missing or invalid gain parameter '${gain}'"
		return 1
	fi

	# Check min/max values.
	min="${SPIAMP_IDX_MIN}"
	max="${SPIAMP_IDX_MAX}"
	if [ "${idx}" -lt "${min}" ] || [ "${idx}" -gt "${max}" ]; then
		log_err "${f}" "Invalid amplifier '${idx}':"
		log_err "${f}" "Must be between ${min} and ${max}"
		return 1
	fi
	min="${SPIAMP_GAIN_MIN}"
	max="${SPIAMP_GAIN_MAX_FORCE}"
	if [ "${gain}" -lt "${min}" ] || [ "${gain}" -gt "${max}" ]; then
		log_err "${f}" "Invalid gain '${gain}':"
		log_err "${f}" "Must be between ${min} and ${max}"
		return 1
	fi
	max="${SPIAMP_GAIN_MAX}"
	if [ "${gain}" -gt "${max}" ] && [ "${force}" != "force" ]; then
		log_err "${f}" "Setting gain above '${max}' require 'force' param"
		return 1
	fi

	# Build the spidev_test command to use and issue it.
	# Convert gain in dB to gain register value and enable the Gain Update on
	# ZeroCrossing flags. The amplifier index matches the SPI chip select.
	gain_val="$((${gain} / 3))"
	gain="$((${gain_val} * 3))"
	flags="0x30"
	spi_cmd="spidev_test -D /dev/spidev${SPIAMP_BUS}.${idx} -p"
	spi_arg=$(printf "\\\x%02x\\\x%02x" "${gain_val}" "${flags}")
	res=$(${spi_cmd} "${spi_arg}" 2>&1) || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Set ${idx} = ${gain}dB: spidev_test FAILED ($ret):"
		log_err "${f}" "${res}"
		return ${ret}
	fi

	log_inf "${f}" "Set ${idx} = ${gain}dB: OK"
	return 0
}

# SPI Amplifiers: Set gain on amplifier (by name).
#
# Parameters:
#  $1: The amplifier name, one of SPIAMP_NAME[] entries.
#  $2: The gain value in dB. Valid values are from 0 to 60, in 3dB steps.
#   Anything above 18 requires $3 set to 'force', otherwise rejected.
#   The amplifiers only accept gain in 3dB multiples, so any value specified
#   that is not a multiple is rounded down to the previous multiple.
#  $3: [opt] Force indicator, required to be set to 'force' for setting gain
#   values larger than 18dB.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
spiamp_gain_set() {
	local name="${1}"
	local gain="${2}"
	local force="${3}"

	local idx

	local ret=0
	local f="${FUNCNAME[0]}"

	# Find the amplifier and get its index.
	_spiamp_find "${name}" idx || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't find the '${name}' amplifier {${ret}}"
		return ${ret}
	fi

	# Set the amplifier's gain.
	_spiamp_gain_set "${idx}" "${gain}" "${force}" || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Set '${name}' gain to ${gain}dB: FAILED (${ret})"
		return ${ret}
	fi

	log_inf "${f}" "Set '${name}' gain to ${gain}dB: OK"
	return 0
}
