#!/bin/sh
# some basic functions to control the TAS-2770/TAS-6421 amplifiers
# from user space when the tas2770/tas6421 is not compiled in, or not working
# Requires i2c-tools to be installed
#
# The amplifier needs to be initialized before doing anything else
# Sequence:
# 1. Source this file. It will make the functions accessible, and prepare
#    the registers from page 0, book 0 to be accessible for writing
# 2. reset the amplifier (call reset_amp)
# 3. call the init_amp function with the start volume as parameter.
#    If not specified, it sets the minimum volume (11 dBV)
# 4. Adjust volume separately (set_pcm_vol and set_tas2771_vol) and finally start the amp (power_on_amp)
# 5. 3 and 4 can be done in one step calling init_start_tas2770
#
# Notes:
# 1. Once the amplifier is initialized, you can change the volume on the fly
# 2. When audio is stopped, and there is nothing to keep the i2s interface
#    "busy" (ex like playing silent loop), so tas-2771 amp has to be initialized
#    again the next time the audio starts. TAS-6421 is not affected by that

# Checking Board HW_Type
# BARIX_M400                54
# BARIX_TPA                 66
# USSI_ENCOMP_CONTROLLER    68
# BARIX_MPA                 69

HW_TYPE=$(/usr/bin/qiba-spi-get-production-info.sh -w)
if [ "$HW_TYPE" = "54" ]; then
    /usr/bin/logger "BARIX_M400 board (54)"
else
    if [ "$HW_TYPE" = "66" ]; then
        /usr/bin/logger "BARIX_TPA board (66)"
        CHIP_ADDR="0x41"
    else
        if [ "$HW_TYPE" = "68" ]; then
            /usr/bin/logger "USSI_ENCOMP_CONTROLLER board (68)"
        else
            if [ "$HW_TYPE" = "69" ]; then
                /usr/bin/logger "BARIX_MPA board (69)"
                CHIP_ADDR="0x6a"
                . /usr/bin/poe-detect.sh
            else
                /usr/bin/logger "Unexpected HW_Type: ($HW_Type)"
            fi
        fi
    fi
fi

/usr/bin/logger "Setting digital audio amplifier"

SET2C=$(which i2cset)
SET2C+=" -y 0"
GET2C=$(which i2cget)
GET2C+=" -y 0"

SET2C+=" $CHIP_ADDR"
GET2C+=" $CHIP_ADDR"


# set TAS2770/TAS6421 register
set_reg() {
    echo "Executing $SET2C $1 $2"
    eval "$SET2C $1 $2"
}

get_reg() {
    ret=$(eval "$GET2C $1")
    echo "$ret"
}

set_tas2771_vol() {
    vol=$1
    echo "Setting TAS2771 Amplifier Volume: $vol"
    set_reg 0x03 "$vol"
}

# pass the PCM volume from 0dB to -100dB in steps of 0.5 dB
# anything below -100 dB mutes it
# ex. call: set_pcm_vol -21.5
set_tas2770_pcm_vol() {
    pcm_vol=$(awk -v val=$1 'function abs(v) {return v < 0 ? -v : v} BEGIN{printf("0x%02x", abs(val)*2)}')
    echo "Setting PCM Volume: $1 to $pcm_vol"
    set_reg 0x05 "$pcm_vol"
}

# changes vol between 24 dB and -100 dB, with 0,5 dB step
# 0dB is at 0xcf
# anything below 0x07 (-100dB) mutes the amp
set_tas6421_pcm_vol() {
    pcm_vol=$(awk -v val=$1 'BEGIN{printf("0x%02x", (val*2)+207)}')
    echo "Setting TAS6421 PCM Volume: $1 to $pcm_vol"
    set_reg 0x05 "$pcm_vol"
}

set_pcm_vol() {
    echo "Requested volume: $1"
    if [ "$HW_TYPE" = "66" ]; then
        set_tas2770_pcm_vol "$1"
    fi
    if [ "$HW_TYPE" = "69" ]; then
        set_tas6421_pcm_vol "$1"
    fi
}

get_pcm_vol() {
    raw_hex_vol=$(get_reg "0x05")
    raw_vol=$(printf "%d" "$raw_hex_vol")
    ret_pcm_vol="0 dB"

    if [ "$HW_TYPE" = "66" ]; then
        ret_pcm_vol=$(awk -v val=$raw_vol 'function calc(v) {return v = 0 ? 0 : -v} BEGIN{printf("%.1f dB", calc(val)/2)}')
    fi
    if [ "$HW_TYPE" = "69" ]; then
        ret_pcm_vol=$(awk -v val=$raw_vol 'BEGIN{printf("%.1fdB", (val-207)/2)}')
    fi
    echo -n "$ret_pcm_vol"
}

power_on_amp() {
    # powering up the amp
    if [ "$HW_TYPE" == "66" ]; then
        set_reg 0x02 0x0c
        echo "TAS2770 amplifier powered on!"
    fi

    if [ "$HW_TYPE" == "69" ]; then
        set_reg 0x04 0x15
        echo "TAS6421 amplifier powered on!"
    fi
}

init_start_tas2770() {
    init_tas2770 "$1"
    power_on_amp
}

init_amp() {
    if [ "$HW_TYPE" = "66" ]; then
        init_tas2770 "$1"
    elif [ "$HW_TYPE" = "69" ]; then
        init_tas6421 "$1" "$2"
    fi
}

init_tas2770() {
    # setting registers
    #set_reg 0x03 0x14
    set_reg 0x04 0x00
    #mute the PCM Volume at start
    set_reg 0x05 0xff

    set_reg 0x07 0x02
    set_reg 0x08 0x40
    set_reg 0x0a 0x37
    set_reg 0x0b 0x03
    set_reg 0x0c 0x10
    set_reg 0x0e 0xf3
    set_reg 0x15 0x15
    set_reg 0x18 0x4b
    set_reg 0x1a 0x10
    set_reg 0x32 0x80
    set_reg 0x3c 0x18
    set_reg 0x7e 0x19
    #setting amplifier volume to intermediate value (18.0 dBV)
    set_tas2771_vol 0x00
    # set pcm_volume if requested
    if [ "$1" != "" ]; then
        set_pcm_vol "$1"
    fi

    sleep 0.002
    echo "TAS2770 I2C init DONE !"
}


init_tas6421() {
    echo "init_tas6421 $1 $2"
    local vol="$1"
    local gain="$2"
    local gain_reg=([0]=0x30 [1]=0x31 [2]=0x32 [3]=0x33)

    # setting registers
    set_reg 0x00 0x00  # default on reset
    set_reg 0x01 ${gain_reg[$(echo "${gain} - 1" | bc -l)]}
    set_reg 0x02 0x66  # set 128x FSCLK
    set_reg 0x03 0x04  # default on reset

    set_reg 0x05 0x00  #mute the PCM Volume at start

    set_reg 0x09 0x00  # default on reset
    set_reg 0x0a 0x11  # default on reset
    #set_reg 0x0c 0x00  # default on reset
    #set_reg 0x0e 0x00  # default on reset
    # 0x0f is state register, to read only. At reset must be 0x40
    # set_reg 0x0f 0x40
    set_reg 0x14 0x00  # default on reset
    set_reg 0x15 0x00  # default on reset
    set_reg 0x16 0x00  # default on reset
    #set_reg 0x17 0x00  # default on reset
    #set_reg 0x1b 0x00  # default on reset
    #set_reg 0x1c 0x00  # default on reset
    set_reg 0x21 0x00  # default on reset
    set_reg 0x22 0x01  # default on reset
    set_reg 0x23 0x14  # default on reset
    set_reg 0x24 0x00  # default on reset

    # set pcm_volume
    set_pcm_vol "$vol"

    sleep 0.002
    echo "AMP I2C init DONE !"
}


# prepare the i2c regs to write
# We mostly write to page 0, book 0, so ignore
# page 1 for now
prepare_tas2770_regs() {
    # set page 0
    set_reg 0x00 0x00
    # set book 0
    set_reg 0x7f 0x00
    # set again page 0
    set_reg 0x00 0x00
}

prepare_regs() {
    if [ "$HW_TYPE" = "66" ]; then
        prepare_tas2770_regs
    fi
}

# reset the TAS2770 device
# always call this function after setting
# the page to write and before the init
reset_amp() {
    if [ "$HW_TYPE" = "66" ]; then
        set_reg 0x01 0x01
    fi
    if [ "$HW_TYPE" = "69" ]; then
        set_reg 0x00 0x80
    fi
}

setup_amp(){
    if [ -z "$1" -o -z "$2" ]; then
        echo "Invalid arguments for setup_amp."
    return
    fi
    local vol="$1"
    local gain="$2"

    prepare_regs
    reset_amp
    sleep 1

    echo "HW_TYPE: $HW_TYPE"

    sleep 0.002

    init_amp "$vol" "$gain"

    set_pcm_vol "$vol"
    sleep 0.002

    # powering up the amp
    power_on_amp
}

tas6421_get_impedance() {
    # Set the output channel into the Hi-Z state
    set_reg 0x04 0x55
    # Set the AC_DIAGS_LOOPBACK bit (bit 7 in register 0x16) to 0
    # use I= 19mA
    set_reg 0x16 0x40
    # Apply a full-scale input signal from the DSP for the tested channel
    # with the desired frequency (recommended 10 khz to 20 khz)
    /usr/bin/aplay -Dplug:digital_out /usr/share/sounds/19kH_stereo_5sec.wav &
    #  Set the device into the AC diagnostic mode (set bit 3 in register 0x15 to 1)
    set_reg 0x15 0x08
    #wait Channel reporting register indicates the status changing:
    # From "AC Diagnostic Mode" TO "Hi-Z" state
    STATUS=$(get_reg "0x0F")
    while [ "$STATUS" != "0x40" ]
    do
        echo "Channel Reporting Register:$STATUS"
        STATUS=$(get_reg "0x0F")
        echo $STATUS
    done
    CH_IMP=$(get_reg "0x17")
    echo "Channel Impedance: $CH_IMP"
    CH_IMP_DEC=$(echo $(($CH_IMP)))
    RES=$(echo "(${CH_IMP_DEC}*2.371)/19" | bc -l)
    echo "Speaker Impedance: $RES ohm"
    echo "speaker.impedance=$RES" > /tmp/speaker
    killall -9 aplay
}

tas6421_set_gain() {
    # Worst case for maximum power consumed. Considering 4W for IPAM400, 5W for USB HUB,
    # 0.406W for i2c io expander, 0.0805W for headphone amp and 0.55W for buck converter.
    local max_power_consumed=10.04

    # Table relating the output power with the load impedance and gain
    #                       gain  impedance  power
    local out_power_tbl=([0]="1       2      14.0625"
                         [1]="1       4      7.03125"
                         [2]="1       8      3.515625"
                         [3]="2       2      56.25"
                         [4]="2       4      28.125"
                         [5]="2       8      14.0625"
                         [6]="3       2      110.25"
                         [7]="3       4      55.125"
                         [8]="3       8      27.5625")
    local poe_power=$1

    if [ -z "$poe_power" ]; then
        echo "No PoE power detected!"
        return
    fi

    # Get speaker impedance
    tas6421_get_impedance
    local impedance=0
    local Speaker_UCI_Cfg=$(/usr/bin/uci get speaker.speaker.config)
    
    if [ "${Speaker_UCI_Cfg}" = "" ] || [ "${Speaker_UCI_Cfg}" = "auto" ]; then
    	if [ -f "/tmp/speaker" ]; then
    		impedance=$(cat /tmp/speaker | grep "speaker.impedance" | sed -e 's/speaker.impedance\s*=\s*//')
	    	impedance=$(printf %0.0f ${impedance})
	fi
    else
	impedance=${Speaker_UCI_Cfg}
    fi

    local remaining_power=$(echo "${poe_power}-${max_power_consumed}" | bc -l)
    echo "Remaining power for amp: ${remaining_power}W"

    # Find the impedance on the table, compare to the output power and extract the gain
    gain=1
    for i in "${out_power_tbl[@]}"
    do
       imp=$(echo ${i} | cut -d" " -f2)
       power=$(echo ${i} | cut -d" " -f3)
       echo "imp ${imp} power ${power}"
       if [ $(echo "${impedance} == ${imp}" | bc -l) -eq 1 ]; then
           if [ $(echo "${remaining_power} >= ${power}" | bc -l) -eq 1 ]; then
               gain=$(echo ${i} | cut -d" " -f1)
           else
               break
           fi
       fi
    done
    
    echo "$gain" > /tmp/poe_amp_gain
    setup_amp 0 "$gain"
}

set_amp_gain(){
    local PoE_UCI_Cfg=$(/usr/bin/uci get poe.poe.config)
    local power=12.95

    local poe_tbl=([0]="poe       12.95"
                   [1]="poe+      25.50"
                   [2]="poe3      51"
                   [3]="poe4      71")

    if [ "$PoE_UCI_Cfg" = "" ] || [ "$PoE_UCI_Cfg" = "auto" ]; then
        if [ -f "/tmp/poe-detect" ]; then
            power=$(cat /tmp/poe-detect | grep "poe.power" | sed -e 's/poe.power\s*=\s*//')
        fi
    else
        for i in "${poe_tbl[@]}"
        do
            poe_cfg=$(echo ${i} | cut -d" " -f1)
            if [ ${PoE_UCI_Cfg} = ${poe_cfg} ]; then
                power=$(echo ${i} | cut -d" " -f2)
                break
            fi
        done
    fi

    tas6421_set_gain "$power"  
}

# ---------------------------------------------------------------
if [ "$HW_TYPE" = "66" ]; then
    /usr/bin/qiba-aplay-loop.sh &
fi

if [ "$HW_TYPE" = "66" ] || [ "$HW_TYPE" = "69" ]; then
    setup_amp 0 1
fi

if [ "$HW_TYPE" = "69" ]; then
    set_amp_gain
fi

if [ "$HW_TYPE" != "66" ]; then
    /usr/bin/qiba-aplay-loop.sh &
fi

